/** @file ilg_cpp_helpers.h
 *  @brief Заголовочный файл SDK Guard с классами-помощниками C++.
 *
 *  Классы-обертки для функций библиотеки из ilguard.h, и
 *  вспомогательными функциями для разработчиков.
 */
#pragma once

#include <chrono>
#include <exception>
#include <string>
#include <vector>

#include "ilguard.h"

namespace ilg {

class CConverter;  // forward declaration

/**
 * @brief Класс исключения SDK Guard.
 * @sa @ref ilg_status
 */
class CILGException : public std::exception {
public:
    /**
     * @brief Конструктор из кода возврата SDK.
     *
     * @param nCode Код возврата SDK.
     */
    CILGException(ilg_status nCode);

    /// Возвращает описание ошибки
    virtual const char* what() const _GLIBCXX_TXN_SAFE_DYN _GLIBCXX_NOTHROW;

public:
    ilg_status m_nCode;  ///< Код возврата SDK
};

/**
 * @brief Бросает исключение если код возврата является ошибкой.
 *
 * Проверяет код возврата SDK и если код соответствует ошибке вызывает исключение CILGException.
 *
 * @param[in]  nCode Код возврата.
 * @exception  CILGException Код возврата не равен `ILG_OK`.
 */
void ILGCheck(ilg_status nCode);

/**
 * @brief Ключ контроллера.
 *
 * Это класс-обертка для ilg_key_number.
 */
class CKeyNumber : public ilg_key_number {
public:
    /// Конструктор по умолчанию
    CKeyNumber();

    /**
     * @brief Конструктор из ilg_key_number.
     *
     * @param[in] other Значение ilg_key_number.
     */
    CKeyNumber(const ilg_key_number& other);

    /**
     * @brief Конструктор номера Em-Marine.
     *
     * @param[in] nEmSeries Серия Em-Marine.
     * @param[in] nEmNumber Номер Em-Marine.
     * @param[in] nFacility Код производителя Em-Marine.
     */
    CKeyNumber(uint8_t nEmSeries, uint16_t nEmNumber, uint32_t nFacility = 0);

    /**
     * @brief Конструктор из байт данных.
     *
     * @param[in] pData Данные.
     * @param[in] nSize Размер данных.
     */
    CKeyNumber(const uint8_t* pData, size_t nSize);

    /**
     * @brief Оператор присвоения номера ilg_key_number.
     *
     * @param[in] other Значение номера.
     */
    const CKeyNumber& operator=(const ilg_key_number& other) {
        ilg_key_number::nData = other.nData;
        return *this;
    }

    /** Обнуляет данные ключа. */
    void Clear();

    /**
     * @brief Устанавливает байты номера ключа.
     *
     * @param[in] pData Данные.
     * @param[in] nSize Размер данных.
     */
    void Assign(const uint8_t* pData, size_t nSize);
};

/**
 * @brief Класс дескриптора SDK.
 *
 * Это класс-обертка для дескриптора SDK (ilg_handle).
 */
class CILGHandle {
    friend class CILG;
    friend class CConverterSearch;
    friend class CConverter;
    friend class CController;

public:
    /// Конструктор по умолчанию.
    CILGHandle();

    /**
     * @brief Конструктор из дескриптора SDK.
     *
     * @param[in] h Значение дескриптора.
     */
    CILGHandle(ilg_handle h);

    /// Запрещаем копирование этого класса (т.к. он содержит указатель = дескриптор).
    CILGHandle(const CILGHandle&) = delete;

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой дескриптор.
     */
    CILGHandle(CILGHandle&& other);

    /// Деструктор
    virtual ~CILGHandle();

    /// Запрещаем копирование этого класса (т.к. он содержит указатель = дескриптор).
    CILGHandle& operator=(const CILGHandle&) = delete;

    /**
     * @brief Оператор перемещения.
     *
     * @param[in out] other Другой дескриптор.
     */
    CILGHandle& operator=(CILGHandle&& other);

    /// Оператор приведения типа к ilg_handle.
    operator ilg_handle() const;

    /// Проверяет дескриптор на валидность.
    explicit operator bool() const;

    /**
     * @brief Обменивается значением с другим дескриптором.
     *
     * @param[in] other Другой дескриптор.
     */
    void Swap(CILGHandle& other) noexcept;

    /// Возвращает дескриптор SDK.
    ilg_handle Get() const;

    /**
     * @brief Закрывает дескриптор SDK.
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    void Close();

    /**
     * @brief Присоединяет дескриптор к этому объекту.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    void Attach(ilg_handle h);

    /**
     * @brief Отсоединяет дескриптор от этого объекту.
     *
     * @return Значение дескриптора SDK.
     */
    ilg_handle Detach();

protected:
    ilg_handle m_h;  ///< Дескриптор SDK
};

/**
 * @brief Класс команды SDK.
 *
 * Класс для управления асинхронными запросом.
 */
class CAsyncCommand : public CILGHandle {
public:
    /// Конструктор по умолчанию.
    CAsyncCommand();

    /**
     * @brief Конструктор класса из дескриптора команды.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    CAsyncCommand(ilg_handle h);

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой дескриптор команды.
     */
    CAsyncCommand(CAsyncCommand&& other);

    /// Деструктор
    virtual ~CAsyncCommand();

    /// Оператор перемещения
    CAsyncCommand& operator=(CAsyncCommand&& other);

    /**
     * @brief Отменяет команду. Устанавливает статус `ILG_E_ABORT`.
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void Cancel();

    /**
     * @brief Возвращает состояние команды.
     *
     * @return Состояние команды: =`ILG_E_PENDING` команда ещё выполняется, иначе - завершена.
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline ilg_status GetStatus() const;

    /**
     * @brief Возвращает состояние прогресса выполнения команды.
     *
     * @param[out] nCurrent  Текущий шаг.
     * @param[out] nTotal    Всего шагов.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void GetProgress(size_t& nCurrent, size_t& nTotal) const;
};

inline void CAsyncCommand::Cancel() {
    ILGCheck(ilg_command_cancel(m_h));
}

inline ilg_status CAsyncCommand::GetStatus() const {
    ilg_status res;
    ILGCheck(ilg_command_get_status(m_h, &res));
    return res;
}

inline void CAsyncCommand::GetProgress(size_t& nCurrent, size_t& nTotal) const {
    ILGCheck(ilg_command_get_progress(m_h, &nCurrent, &nTotal));
}

/**
 * @brief Класс поиска конвертеров.
 *
 * Класс-обёртка для дескриптора поиска конвертеров.
 */
class CConverterSearch : public CILGHandle {
public:
    /// Конструктор по умолчанию.
    CConverterSearch();

    /**
     * @brief Конструктор класса из дескриптора поиска конвертеров.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    CConverterSearch(ilg_handle h);

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой поиск конвертеров.
     */
    CConverterSearch(CConverterSearch&& other);

    /// Деструктор
    virtual ~CConverterSearch();

    /// Оператор перемещения
    CConverterSearch& operator=(CConverterSearch&& other);

    /**
     * @brief Устанавливает функцию обратного вызова для уведомлений поиска конвертеров.
     *
     * Устанавливает функцию для получения сообщений от дескриптора поиска конвертеров.
     *
     * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать при
     * возникновении события поиска конвертеров.
     * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе будет ошибка
     * ``ILG_E_BLOCKING_CALL_NOT_ALLOWED``.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetMessageCallback(ilg_search_message_callback pCallback,
                                   void* pUserData = nullptr);

    /**
     * @brief Включает/выключает очередь сообщений.
     *
     * Эта функция устанавливает/снимает флаг в дескрипторе поиска.
     * Очередь сообщений предназначена для синхронизации обработки сообщений.
     *
     * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
     * вызывается функция обратного вызова, установленная функцией @ref SetMessageCallback, из
     * которой посылается сигнал потоку, обрабатывающему сообщения, этот поток при получении сигнала
     * циклично вызывает @ref GetMessage, чтобы получить и обработать все сообщения.
     *
     * @param[in] fEnable  `true`, включить очередь, иначе - выключить.
     *
     * @warning Если не извлекать сообщения из очереди функцией @ref GetMessage, то
     * она будет расти пока не закончится память.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void EnableMessageQueue(bool fEnable = true);

    /**
     * @brief Извлекает следующее сообщение из очереди.
     *
     * @param[out] nMsg     Тип сообщения.
     * @param[out] pMsgData Указатель на данные пользователя, которые были установлены функцией
     * @ref SetMessageCallback.
     *
     * @return true если сообщение успешно извлечено, иначе - очередь пуста.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetMessage(ilg_search_msg& nMsg, const void*& pMsgData) const;

    /**
     * @brief Устанавливает параметры поиска считывателей.
     *
     * @remark Параметры поиска считывателей общие для всех дескрипторов.
     * Параметры не теряются при закрытии всех дескрипторов поиска.
     *
     * @param[in] rOptions  Опции поиска считывателей.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetOptions(const ilg_search_options& rOptions);

    /**
     * @brief Возвращает параметры поиска считывателей.
     *
     * @param[out] rOptions  Опции поиска считывателей.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void GetOptions(ilg_search_options& rOptions) const;

    /**
     * @brief Устанавливает список портов для прослушки конвертеров к режиме "Клиент".
     *
     * Устанавливает список TCP-портов, к которым будут подключаться IP-конвертеры
     * в режиме "Клиент".
     *
     * @remark Список портов общий для всех дескрипторов.
     *
     * @param[in] pPorts  Массив портов. Если =`nullptr`, то очищает список.
     * @param[in] nCount  Количество элементов массива портов.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_INVALIDARG`,
     * `ILG_E_OUTOFMEMORY`.
     */
    inline void SetListenPorts(const uint16_t* pPorts, size_t nCount);

    /**
     * @brief Возвращает список портов для прослушки конвертеров к режиме "Клиент".
     *
     * Возвращает список TCP-портов, который был установлен функцией @ref SetListenPorts.
     *
     * @remark Список портов общий для всех дескрипторов.
     *
     * @param[in out] oPorts  Список портов.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_OUTOFMEMORY`.
     */
    void GetListenPorts(std::vector<uint16_t>& oPorts) const;

    /**
     * @brief Возвращает состояние Tcp-порта, открытого для прослушки конвертеров в режиме "Клиент".
     *
     * @param[in]  nTcpPort  Номер TCP порта.
     *
     * @remark Чтобы открыть порт для прослушки нужно добавить его в список портов с помощью функции
     * @ref ilg_search_set_listen_ports, и включить поиск конвертеров в режиме "Клиент" с помощью
     * @ref ilg_search_set_options (установить `pOptions->nConverterTypes |= ILG_CONVERTER_CLIENT`).
     *
     * @return Состояние порта: `ILG_OK` порт открыт успешно, `ILG_E_PENDING` порт в процессе
     * открытия или его нет в списке портов для прослушки, иначе не удалось открыть.
     */
    inline ilg_status GetListenStatus(uint16_t nTcpPort) const;

    /**
     * @brief Ищет считыватели.
     *
     * @param[in] fReset  =`true` очищает список найденных считывателей перед началом поиска.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * считывателей.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`,
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`, `ILG_E_OUT_OF_RESOURCES`, `ILG_E_OUTOFMEMORY`.
     */
    inline void Scan(bool fReset = false);

    /**
     * @brief Запускает асинхронную команду поиска считывателей.
     *
     * @param[in]  fReset =`true` очистить список найденных перед поиском.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`,
     * `ILG_E_OUT_OF_RESOURCES`, `ILG_E_OUTOFMEMORY`.
     */
    inline CAsyncCommand Begin_Scan(bool fReset = false);

    /**
     * @brief Возвращает количество найденных считывателей.
     *
     * @warning Эта функция копирует список из потока поиска, поэтому её лучше не вызывать в цикле.
     *
     * @return Количество найденных считывателей.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_OUTOFMEMORY`.
     */
    inline size_t GetConverterCount() const;

    /**
     * @brief Возвращает информацию о найденном считывателе.
     *
     * @param[in]  nIdx    Позиция в списке найденных считывателей.
     * @param[out] rInfo   Информация о считывателе.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_BOUNDS`.
     */
    inline void GetConverterInfo(size_t nIdx, ilg_converter_info& rInfo) const;

    /**
     * @brief Включает/выключает авто поиск считывателей.
     *
     * @param[in]  fEnable =`true`, включает авто поиск, иначе - выключает.
     * @param[in]  fWait   =`true`, ждёт полного завершения команды, иначе только устанавливает
     * флаг.
     *
     * @remark Если @p fWait =`true`, то функция не возвращает управление пока ждёт выполнение
     * команды в потоке поиска считывателей.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`,
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`, `ILG_E_OUTOFMEMORY`, `ILG_E_OUT_OF_RESOURCES`.
     */
    inline void SetAutoScan(bool fEnable = true, bool fWait = true);

    /**
     * @brief Запускает асинхронную команду вкл/выкл режим авто поиска считывателей.
     *
     * @param[in]  fEnable =`true` включает поиск в реальном времени, иначе - выключает.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_OUTOFMEMORY`,
     * `ILG_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_SetAutoScan(bool fEnable);

    /**
     * @brief Возвращает флаг авто поиска считывателей.
     *
     * @return true, авто поиск включен, иначе - выключен.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetAutoScan() const;

    /**
     * @brief Открывает порт и возвращает его дескриптор.
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     * @param[out] pInfo       Информация о считывателе. Может быть равно `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * считывателей.
     *
     * @return Системный дескриптор порта.
     *
     * @exception CILGException если произошла ошибка `ILG_E_INVALIDARG`, `ILG_E_HANDLE`,
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`, `ILG_E_OUTOFMEMORY`, `ILG_E_OUT_OF_RESOURCES`,
     * `ILG_E_PORT_ACCESS_DENIED`, `ILG_E_PORT_NOT_EXIST`, `ILG_E_PORT_OPEN_FAILED`,
     * `ILG_E_CONNECTION_ERROR`, `ILG_E_NOTIMPL`.
     */
    inline int OpenPort(ilg_port_type nPortType, const char* pszPortName,
                        ilg_converter_info* pInfo = nullptr);

    /**
     * @brief Запускает асинхронную команду открытия порта.
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка `ILG_E_INVALIDARG`, `ILG_E_HANDLE`,
     * `ILG_E_OUTOFMEMORY`, `ILG_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_OpenPort(ilg_port_type nPortType, const char* pszPortName);

    /**
     * @brief Возвращает результат открытия порта.
     *
     * @param[in]  hCommand Дескриптор команды.
     * @param[out] PortFD   Системный дескриптор порта.
     * @param[out] rInfo    Информация о конвертере (если известно).
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_OpenPort(ilg_handle hCommand, int& PortFD, ilg_converter_info& rInfo);

    /**
     * @brief Закрывает порт
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     * @param[in]  hPort       Системный дескриптор порта.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * считывателей.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`,
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`, `ILG_E_OUTOFMEMORY`, `ILG_E_OUT_OF_RESOURCES`.
     */
    inline void ClosePort(ilg_port_type nPortType, const char* pszPortName, int hPort);

    /**
     * @brief Запускает асинхронную команду закрытия порта.
     *
     * @param[in]  nPortType   Тип порта
     * @param[in]  pszPortName Имя порта
     * @param[in]  hPortFD     Системный дескриптор порта.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_OUTOFMEMORY`,
     * `ILG_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_ClosePort(ilg_port_type nPortType, const char* pszPortName,
                                         int hPortFD);

    /**
     * @brief Клонирует дескриптор поиска конвертеров.
     * @return Дескриптор поиска конвертеров.
     */
    inline CConverterSearch Clone() const;
};

inline void CConverterSearch::SetMessageCallback(ilg_search_message_callback pCallback,
                                                 void* pUserData) {
    ILGCheck(ilg_search_set_message_callback(m_h, pCallback, pUserData));
}

inline void CConverterSearch::EnableMessageQueue(bool fEnable) {
    ILGCheck(ilg_search_enable_message_queue(m_h, (ilg_bool)fEnable));
}

inline bool CConverterSearch::GetMessage(ilg_search_msg& nMsg, const void*& pMsgData) const {
    ilg_bool f;
    ILGCheck(ilg_search_get_message(m_h, &nMsg, &pMsgData, &f));
    return f;
}

inline void CConverterSearch::SetOptions(const ilg_search_options& rOptions) {
    ILGCheck(ilg_search_set_options(m_h, &rOptions));
}

inline void CConverterSearch::GetOptions(ilg_search_options& rOptions) const {
    ILGCheck(ilg_search_get_options(m_h, &rOptions));
}

inline void CConverterSearch::SetListenPorts(const uint16_t* pPorts, size_t nCount) {
    ILGCheck(ilg_search_set_listen_ports(m_h, pPorts, nCount));
}

ilg_status CConverterSearch::GetListenStatus(uint16_t nTcpPort) const {
    ilg_status res;
    ILGCheck(ilg_search_get_listen_status(m_h, nTcpPort, &res));
    return res;
}

inline void CConverterSearch::Scan(bool fReset) {
    ILGCheck(ilg_search_scan(m_h, (ilg_bool)fReset));
}

inline size_t CConverterSearch::GetConverterCount() const {
    size_t res;
    ILGCheck(ilg_search_get_converter_count(m_h, &res));
    return res;
}

inline void CConverterSearch::GetConverterInfo(size_t nIdx, ilg_converter_info& rInfo) const {
    ILGCheck(ilg_search_get_converter_info(m_h, nIdx, &rInfo));
}

inline void CConverterSearch::SetAutoScan(bool fEnable, bool fWait) {
    ILGCheck(ilg_search_set_auto_scan(m_h, (ilg_bool)fEnable, (ilg_bool)fWait));
}

inline bool CConverterSearch::GetAutoScan() const {
    ilg_bool f;
    ILGCheck(ilg_search_get_auto_scan(m_h, &f));
    return static_cast<bool>(f);
}

inline int CConverterSearch::OpenPort(ilg_port_type nPortType, const char* pszPortName,
                                      ilg_converter_info* pInfo) {
    int fd;
    ILGCheck(ilg_search_open_port(m_h, nPortType, pszPortName, pInfo, &fd));
    return fd;
}

inline void CConverterSearch::ClosePort(ilg_port_type nPortType, const char* pszPortName,
                                        int hPort) {
    ILGCheck(ilg_search_close_port(m_h, nPortType, pszPortName, hPort));
}

inline CAsyncCommand CConverterSearch::Begin_Scan(bool fReset) {
    CAsyncCommand res;
    ILGCheck(ilg_search_begin_scan(m_h, (ilg_bool)fReset, &res.m_h));
    return res;
}

inline CAsyncCommand CConverterSearch::Begin_SetAutoScan(bool fEnable) {
    CAsyncCommand res;
    ILGCheck(ilg_search_begin_set_auto_scan(m_h, (ilg_bool)fEnable, &res.m_h));
    return res;
}

inline CAsyncCommand CConverterSearch::Begin_OpenPort(ilg_port_type nPortType,
                                                      const char* pszPortName) {
    CAsyncCommand res;
    ILGCheck(ilg_search_begin_open_port(m_h, nPortType, pszPortName, &res.m_h));
    return res;
}

inline void CConverterSearch::End_OpenPort(ilg_handle hCommand, int& PortFD,
                                           ilg_converter_info& rInfo) {
    ILGCheck(ilg_search_end_open_port(hCommand, &PortFD, &rInfo));
}

inline CAsyncCommand CConverterSearch::Begin_ClosePort(ilg_port_type nPortType,
                                                       const char* pszPortName, int hPortFD) {
    CAsyncCommand res;
    ILGCheck(ilg_search_begin_close_port(m_h, nPortType, pszPortName, hPortFD, &res.m_h));
    return res;
}

inline CConverterSearch CConverterSearch::Clone() const {
    ilg_handle hNewHandle;
    ILGCheck(ilg_clone_handle(m_h, &hNewHandle));
    return CConverterSearch(hNewHandle);
}

/**
 * @brief Класс контроллера.
 *
 * Класс-обёртка для дескриптора контроллера.
 */
class CController : public CILGHandle {
public:
    /// Конструктор по умолчанию.
    CController();

    /**
     * @brief Конструктор класса из дескриптора контроллера.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    CController(ilg_handle h);

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой контроллер.
     */
    CController(CController&& other);

    /// Деструктор
    virtual ~CController();

    /// Оператор перемещения
    CController& operator=(CController&& other);

    /**
     * @brief Устанавливает функцию обратного вызова для уведомлений контроллера.
     *
     * Устанавливает функцию для получения сообщений от дескриптора контроллера.
     *
     * @param[in] pCallback   Указатель на функцию, которую библиотека будет вызывать при
     * возникновении события контроллера.
     * @param[in] pUserData   Указатель на данные пользователя, который будет передаваться в
     * функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе вернёт ошибку
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetMessageCallback(ilg_controller_message_callback pCallback,
                                   void* pUserData = nullptr);

    /**
     * @brief Включает/выключает очередь сообщений.
     *
     * Эта функция устанавливает/снимает флаг в дескрипторе контроллера.
     * Очередь сообщений предназначена для синхронизации обработки сообщений.
     *
     * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
     * вызывается функция обратного вызова, установленная функцией @ref SetMessageCallback, из
     * которой посылается сигнал потоку, обрабатывающему сообщения, этот поток при получении сигнала
     * циклично вызывает @ref GetMessage, чтобы получить и обработать все сообщения.
     *
     * @param[in] fEnable  `true`, включает очередь, иначе - выключает.
     *
     * @warning Если не извлекать сообщения из очереди функцией @ref GetMessage, то она будет расти
     * пока не закончится память.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void EnableMessageQueue(bool fEnable = true);

    /**
     * @brief Извлекает следующее сообщение из очереди.
     *
     * @param[out] nMsg       Тип сообщения.
     * @param[out] pMsgData   Указатель на данные пользователя, которые были установлены функцией
     * @ref SetMessageCallback.
     *
     * @return true если сообщение успешно извлечено, иначе - очередь пуста.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetMessage(ilg_controller_msg& nMsg, const void*& pMsgData) const;

    /**
     * @brief Устанавливает параметры контроллера.
     *
     * @param[in] rOptions  Параметры контроллера.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_INVALIDARG`.
     */
    inline void SetOptions(const ilg_controller_options& rOptions);

    /**
     * @brief Возвращает параметры контроллера.
     *
     * @param[out] rOptions  Параметры контроллера.
     *
     * @exception CILGException если произошла ошибка `ILG_E_HANDLE`, `ILG_E_POINTER`.
     */
    inline void GetOptions(ilg_controller_options& rOptions);

    /**
     * @brief Подключается к контроллеру.
     *
     * @param[in] fReconnect   `true`, отключается перед подключением.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void Connect(bool fReconnect = false);

    /**
     * @brief Запускает асинхронную команду подключения к контроллеру.
     *
     * @param[in]  fReconnect  `true`, переподключиться.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_Connect(bool fReconnect = false);

    /**
     * @brief Отключается от контроллера.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void Disconnect();

    /**
     * @brief Запускает асинхронную команду отключения от контроллера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_Disconnect();

    /**
     * @brief Возвращает состояние подключения к контроллеру.
     *
     * @return Состояние подключения к контроллеру.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline ilg_connection_status GetConnectionStatus() const;

    /**
     * @brief Возвращает информацию о контроллере.
     *
     * @param[out] rInfo  Информация о контроллере.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetControllerInfo(ilg_controller_info& rInfo) const;

    /**
     * @brief Возвращает информацию о конвертере.
     *
     * @param[out] rInfo  Информация о конвертере.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetConverterInfo(ilg_converter_info& rInfo) const;

    /**
     * @brief Создаёт дескриптор конвертера.
     *
     * @return Новый дескриптор конвертера.
     */
    inline CConverter GetConverter() const;

    /**
     * @brief Читает информационные строки контроллера.
     *
     * @return Информационные строки контроллера.
     *
     * @exception CILGException если произошла ошибка.
     */
    std::string ReadLines();

    /**
     * @brief Запускает асинхронную команду чтения инфо строк контроллера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadLines();

    /**
     * @brief Возвращает результат чтения инфо строк контроллера.
     *
     * @param[in]     hCommand  Дескриптор команды, который вернула @ref Begin_ReadLines.
     * @param[out]    sLines    Строки контроллера.
     *
     * @exception CILGException если произошла ошибка.
     */
    static void End_ReadLines(ilg_handle hCommand, std::string& sLines);

    /**
     * @brief Читает времена замка из памяти контроллера.
     *
     * @param[in]  nBankN    Номер банка: =0 вход, =1 выход.
     * @param[out] nOpen     Время открывания замка в 1/10 секунды.
     * @param[out] nLet      Время ожидания открытия двери в 1/10 секунды.
     * @param[out] nMax      Время ожидания закрытия двери в 1/10 секунды.
     * @param[out] fBigTime  `true`, используется формат большого времени 0..65535, иначе 0..255.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadLockTimes(uint8_t nBankN, uint16_t& nOpen, uint16_t& nLet, uint16_t& nMax,
                              bool& fBigTime);

    /**
     * @brief Запускает асинхронную команду чтения времён замка.
     *
     * @param[in]  nBankN  Номер банка: =0 вход, =1 выход.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadLockTimes(uint8_t nBankN);

    /**
     * @brief Возвращает результат чтения времён замка.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadLockTimes.
     * @param[out] nOpen     Время открывания замка в 1/10 секунды.
     * @param[out] nLet      Время ожидания открытия двери в 1/10 секунды.
     * @param[out] nMax      Время ожидания закрытия двери в 1/10 секунды.
     * @param[out] fBigTime  `true`, используется формат большого времени 0..65535, иначе 0..255.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadLockTimes(ilg_handle hCommand, uint16_t& nOpen, uint16_t& nLet,
                                         uint16_t& nMax, bool& fBigTime);

    /**
     * @brief Пишет времена замка в память контроллера.
     *
     * @param[in] nBankN    Номер банка: =0 вход, =1 выход.
     * @param[in] nOpen     Время открывания замка в 1/10 секунды.
     * @param[in] nLet      Время ожидания открытия двери в 1/10 секунды.
     * @param[in] nMax      Время ожидания закрытия двери в 1/10 секунды.
     * @param[in] fBigTime  `true`, используется формат большого времени 0..65535, иначе 0..255.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteLockTimes(uint8_t nBankN, uint16_t nOpen, uint16_t nLet, uint16_t nMax,
                               bool fBigTime);

    /**
     * @brief Запускает асинхронную команду записи времён замка.
     *
     * @param[in] nBankN    Номер банка: =0 вход, =1 выход.
     * @param[in] nOpen     Время открывания замка в 1/10 секунды.
     * @param[in] nLet      Время ожидания открытия двери в 1/10 секунды.
     * @param[in] nMax      Время ожидания закрытия двери в 1/10 секунды.
     * @param[in] fBigTime  `true`, используется формат большого времени 0..65535, иначе 0..255.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteLockTimes(uint8_t nBankN, uint16_t nOpen, uint16_t nLet,
                                              uint16_t nMax, bool fBigTime);

    /**
     * @brief Читает время антипассбэк из контроллера.
     *
     * @param[out] fEnabled  `true`, время задействовано.
     * @param[out] nMinutes  Время антипассбэк в минутах.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadApbTime(bool& fEnabled, uint16_t& nMinutes);

    /**
     * @brief Запускает асинхронную команду чтения времени антипассбэк.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadApbTime();

    /**
     * @brief Возвращает результат чтения времени антипассбэк.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadApbTime.
     * @param[out] fEnabled  `true`, время задействовано.
     * @param[out] nMinutes  Время антипассбэк в минутах.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadApbTime(ilg_handle hCommand, bool& fEnabled, uint16_t& nMinutes);

    /**
     * @brief Пишет время антипассбэк в контроллер.
     *
     * @param[in] fEnabled  `true`, время задействовано.
     * @param[in] nMinutes  Время антипассбэк в минутах.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteApbTime(bool fEnabled, uint16_t nMinutes);

    /**
     * @brief Запускает асинхронную команду записи времён замка.
     *
     * @param[in]  fEnabled  `true`, время задействовано.
     * @param[in]  nMinutes  Время антипассбэк в минутах.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteApbTime(bool fEnabled, uint16_t nMinutes);

    /**
     * @brief Читает временные зоны из контроллера.
     *
     * @param[in]  nBankN  Номер банка: =0 вход, =1 выход.
     * @param[in]  nIdx    Позиция временной зоны в банке.
     * @param[out] pBuf    Буфер для временных зон.
     * @param[in]  nCount  Количество зон, которые нужно прочитать.
     * @param[out] pRead   Количество прочитанных временных зон. Если функция завершилась успешно,
     * то всегда равно @p nCount.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadTimeZones(uint8_t nBankN, size_t nIdx, ilg_time_zone* pBuf, size_t nCount,
                              size_t* pRead = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения временных зон.
     *
     * @param[in] nBankN  Номер банка: =0 вход, =1 выход.
     * @param[in] nIdx    Позиция временной зоны в банке.
     * @param[in] nCount  Количество зон, которые нужно прочитать.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadTimeZones(uint8_t nBankN, size_t nIdx, size_t nCount);

    /**
     * @brief Возвращает результат чтения временных зон.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] pList     Ссылка на список прочитанных временных зон. Ссылка действительна до
     * закрытия дескриптора команды @p hCommand.
     * @param[out] nRead     Количество прочитанных временных зон. Если команда выполнена успешно,
     * то равно запрошенному количеству временных зон.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadTimeZones(ilg_handle hCommand, const ilg_time_zone*& pList,
                                         size_t& nRead);

    /**
     * @brief Пишет временные зоны в контроллер.
     *
     * @param[in]  nBankN    Номер банка: =0 вход, =1 выход.
     * @param[in]  nIdx      Позиция временной зоны в банке.
     * @param[in]  pTZs      Список временных зон.
     * @param[in]  nCount    Количество зон, которые нужно записать.
     * @param[out] pWritten  Количество записанных зон. Если команда выполнена успешно, то равно
     * количеству временных зон @p nCount.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteTimeZones(uint8_t nBankN, size_t nIdx, const ilg_time_zone* pTZs,
                               size_t nCount, size_t* pWritten = nullptr);

    /**
     * @brief Запускает асинхронную команду записи временных зон.
     *
     * @param[in] nBankN  Номер банка: =0 вход, =1 выход.
     * @param[in] nIdx    Позиция временной зоны в банке.
     * @param[in] pTZs    Список временных зон.
     * @param[in] nCount  Количество зон, которые нужно записать.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteTimeZones(uint8_t nBankN, size_t nIdx,
                                              const ilg_time_zone* pTZs, size_t nCount);

    /**
     * @brief Возвращает результат записи временных зон.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_WriteTimeZones.
     * @param[out] nWritten  Количество записанных зон. Если функция выполнена успешно, то равно
     * количеству записываемых временных зон.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_WriteTimeZones(ilg_handle hCommand, size_t& nWritten);

    /**
     * @brief Читает временные зоны для переключения режима контроллера.
     *
     * Читает из памяти контроллера временные зоны для переключения режима контроллера.
     *
     * @param[in]  nIdx    Позиция временной зоны 0..1.
     * @param[out] pBuf    Буфер для временных зон.
     * @param[in]  nCount  Количество считываемых временных зон 1..2.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadModeTimeZones(size_t nIdx, ilg_mode_time_zone* pBuf, size_t nCount);

    /**
     * @brief Запускает асинхронную команду чтения временных зон для переключения режима
     * контроллера.
     *
     * @param[in] nIdx    Позиция временной зоны 0..1.
     * @param[in] nCount  Количество считываемых временных зон 1..2.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadModeTimZones(size_t nIdx, size_t nCount);

    /**
     * @brief Возвращает результат чтения временных зон для переключения режима контроллера.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadModeTimZones.
     * @param[out] pList     Ссылка на список прочитанных временных зон. Ссылка действительна до
     * закрытия дескриптора команды @p hCommand.
     * @param[out] nRead     Количество прочитанных временных зон. Равно количеству запрошенных зон.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadModeTimZones(ilg_handle hCommand, const ilg_mode_time_zone*& pList,
                                            size_t& nRead);

    /**
     * @brief Пишет временные зоны для переключения режима контроллера.
     *
     * Пишет в память контроллера временные зоны для переключения режима контроллера.
     *
     * @param[in] nIdx    Позиция временной зоны 0..1.
     * @param[in] pTZs    Список временных зон.
     * @param[in] nCount  Количество записываемых временных зон 1..2.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteModeTimeZones(size_t nIdx, const ilg_mode_time_zone* pTZs, size_t nCount);

    /**
     * @brief Запускает асинхронную команду записи временных зон для переключения режима
     * контроллера.
     *
     * @param[in] nIdx    Позиция временной зоны 0..1.
     * @param[in] pTZs    Список временных зон.
     * @param[in] nCount  Количество записываемых временных зон 1..2.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteModeTimeZones(size_t nIdx, const ilg_mode_time_zone* pTZs,
                                                  size_t nCount);

    /**
     * @brief Читает ключи из памяти контроллера.
     *
     * @param[in]  nBankN  Номер банка: =0 вход, =1 выход.
     * @param[in]  nIdx    Позиция ключа в банке ключей.
     * @param[out] pBuf    Буфер для ключей.
     * @param[in]  nCount  Количество ключей, которые нужно прочитать.
     * @param[out] pRead   Количество прочитанных ключей. Если функция завершилась успешно, то
     * всегда равно @p nCount.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadKeys(uint8_t nBankN, size_t nIdx, ilg_key* pBuf, size_t nCount,
                         size_t* pRead = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения ключей.
     *
     * @param[in] nBankN  Номер банка: =0 вход, =1 выход.
     * @param[in] nIdx    Позиция ключа в банке ключей.
     * @param[in] nCount  Количество ключей, которые нужно прочитать.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadKeys(uint8_t nBankN, size_t nIdx, size_t nCount);

    /**
     * @brief Возвращает результат чтения ключей.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadKeys.
     * @param[out] pList     Ссылка на массив прочитанных ключей. Ссылка действительна до закрытия
     * дескриптора команды.
     * @param[out] nRead     Количество прочитанных ключей. Если команда выполнена успешно, то равно
     * количеству запрошенных ключей.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadKeys(ilg_handle hCommand, const ilg_key*& pList, size_t& nRead);

    /**
     * @brief Пишет ключи в память контроллера.
     *
     * @param[in]  nBankN     Номер банка: =0 вход, =1 выход.
     * @param[in]  nIdx       Позиция ключа в банке ключей.
     * @param[in]  pKeys      Список ключей.
     * @param[in]  nCount     Количество ключей, которые нужно записать.
     * @param[out] pWritten   Количество записанных ключей. Если функция завершилась успешно, то
     * всегда равно @p nCount.
     * @param[in]  fSelected  `true`, записывать только выделенные ключи с установленным флагом
     * fSelected в @ref ilg_key.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * @warning В режиме `Advanced` конвертера действует лицензия (@see @ref ilg_license_info),
     * которая может ограничивать до какой позиции можно записывать ключи.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteKeys(uint8_t nBankN, size_t nIdx, const ilg_key* pKeys, size_t nCount,
                          size_t* pWritten = nullptr, bool fSelected = false);

    /**
     * @brief Запускает асинхронную команду записи ключей.
     *
     * @param[in] nBankN     Номер банка: =0 вход, =1 выход.
     * @param[in] nIdx       Позиция ключа в банке ключей.
     * @param[in] pKeys      Список ключей.
     * @param[in] nCount     Количество ключей, которые нужно записать.
     * @param[in] fSelected  `true`, записывать только выделенные ключи с установленным флагом
     * ilg_key.fSelected.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteKeys(uint8_t nBankN, size_t nIdx, const ilg_key* pKeys,
                                         size_t nCount, bool fSelected = false);

    /**
     * @brief Возвращает результат записи ключей.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_WriteKeys.
     * @param[out] nWritten  Количество записанных ключей. Если команда выполнена успешно, то равно
     * количеству записываемых ключей.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_WriteKeys(ilg_handle hCommand, size_t& nWritten);

    /**
     * @brief Стирает ключи в памяти контроллера.
     *
     * @param[in]  nBankN   Номер банка: =0 вход, =1 выход.
     * @param[in]  nIdx     Позиция ключа в банке ключей.
     * @param[in]  nCount   Количество ключей, которые нужно стереть.
     * @param[out] pErased  Количество стёртых ключей. Если функция завершилась успешно, то всегда
     * равно @p nCount.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void EraseKeys(uint8_t nBankN, size_t nIdx, size_t nCount, size_t* pErased = nullptr);

    /**
     * @brief Запускает асинхронную команду стирания ключей.
     *
     * @param[in] nBankN  Номер банка: =0 вход, =1 выход.
     * @param[in] nIdx    Позиция ключа в банке ключей.
     * @param[in] nCount  Количество ключей, которые нужно стереть.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_EraseKeys(uint8_t nBankN, size_t nIdx, size_t nCount);

    /**
     * @brief Возвращает результат стирания ключей.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_EraseKeys.
     * @param[out] nErased   Количество стёртых ключей. Если функция завершилась успешно, то равно
     * количеству стираемых ключей.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_EraseKeys(ilg_handle hCommand, size_t& nErased);

    /**
     * @brief Стирает ключи в указанных ячейках.
     *
     * @param[in]  nBankN   Номер банка: =0 вход, =1 выход.
     * @param[in]  pIdxs    Список позиций ключей, сортированный по возрастанию.
     * @param[in]  nCount   Количество ключей, которые нужно стереть.
     * @param[out] pErased  Количество стёртых ключей. Если функция завершилась успешно, то всегда
     * равно @p nCount.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void EraseKeys2(uint8_t nBankN, const size_t* pIdxs, size_t nCount,
                           size_t* pErased = nullptr);

    /**
     * @brief Запускает асинхронную команду стирания ключей в указанных ячейках.
     *
     * @param[in] nBankN  Номер банка: =0 вход, =1 выход.
     * @param[in] pIdxs   Список позиций ключей, сортированный по возрастанию.
     * @param[in] nCount  Количество ключей, которые нужно стереть.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_EraseKeys2(uint8_t nBankN, const size_t* pIdxs, size_t nCount);

    /**
     * @brief Возвращает результат стирания ключей в указанных ячейках.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_EraseKeys2.
     * @param[out] nErased   Количество стёртых ключей. Если функция завершилась успешно, то равно
     * количеству стираемых ключей.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_EraseKeys2(ilg_handle hCommand, size_t& nErased);

    /**
     * @brief Читает верхнюю границу ключей.
     *
     * Читает из контроллера позицию верхней границы ключей, начиная с которой все ячейки стёрты.
     *
     * @param[in] nBankN  Номер банка: =0 вход, =1 выход.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @return Позиция верхней границы ключей. Если равно -1, то в контроллере адрес верхней границы
     * не корректный.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline ssize_t ReadKeyTopIdx(uint8_t nBankN);

    /**
     * @brief Запускает асинхронную команду чтения верхней границы ключей.
     *
     * @param[in] nBankN  Номер банка: =0 вход, =1 выход.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadKeyTopIdx(uint8_t nBankN);

    /**
     * @brief Возвращает результат чтения верхней границы ключей.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadKeyTopIdx.
     * @param[out] nIdx      Позиция верхней границы ключей. Если равно -1, то в контроллере адрес
     * верхней границы не корректный.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadKeyTopIdx(ilg_handle hCommand, ssize_t& nIdx);

    /**
     * @brief Читает события из памяти контроллера.
     *
     * @param[in]  nIdx    Позиция события в банке событий.
     * @param[out] pBuf    Буфер для событий.
     * @param[in]  nCount  Количество событий, которые нужно прочитать.
     * @param[out] pRead   Количество прочитанных событий. Если функция завершилась успешно, то
     * всегда равно @p nCount.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Получить позиции указателя чтения и указателя записи можно с помощью функции @ref
     * ReadRtcParams.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadEvents(size_t nIdx, uint64_t* pBuf, size_t nCount, size_t* pRead = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения событий.

     * @param[in] nIdx    Позиция события в банке событий.
     * @param[in] nCount  Количество событий, которые нужно прочитать.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadEvents(size_t nIdx, size_t nCount);

    /**
     * @brief Возвращает результат чтения событий.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadEvents.
     * @param[out] pList     Ссылка на список прочитанных событий. Ссылка действительна до закрытия
     * дескриптора команды.
     * @param[out] nRead     Количество прочитанных событий. Если функция завершилась успешно, то
     * равно количеству запрошенных событий.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadEvents(ilg_handle hCommand, const uint64_t*& pList, size_t& nRead);

    /**
     * @brief Пишет позицию указателя чтения событий.
     *
     * @param[in] nIdx  Позиция указателя чтения событий.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteEventReadIdx(size_t nIdx);

    /**
     * @brief Запускает асинхронную команду записи указателя чтения событий.
     *
     * @param[in] nIdx  Позиция указателя чтения событий.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteEventReadIdx(size_t nIdx);

    /**
     * @brief Читает параметры RTC из контроллера.
     *
     * @param[out] rParams  Параметры RTC.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadRtcParams(ilg_rtc_params& rParams);

    /**
     * @brief Запускает асинхронную команду чтения параметров RTC.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadRtcParams();

    /**
     * @brief Возвращает результат чтения параметров RTC.
     *
     * @param[in]  hCommand  Дескриптор команды, которую вернула @ref Begin_ReadRtcParams.
     * @param[out] rParams   Параметры RTC.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadRtcParams(ilg_handle hCommand, ilg_rtc_params& rParams);

    /**
     * @brief Возвращает результат авто опроса параметров RTC.
     *
     * @param[out] rParams  Параметры RTC.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetRtcParams(ilg_rtc_params& rParams);

    /**
     * @brief Читает параметры ExtAsk из контроллера.
     *
     * @param[out] rParams  Параметры ExtAsk.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadExtAskParams(ilg_extask_params& rParams);

    /**
     * @brief Запускает асинхронную команду чтения параметров ExtAsk.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadExtAskParams();

    /**
     * @brief Возвращает результат чтения параметров ExtAsk.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadExtAskParams.
     * @param[out] rParams   Параметры ExtAsk.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadExtAskParams(ilg_handle hCommand, ilg_extask_params& rParams);

    /**
     * @brief Возвращает результат авто опроса ExtAsk.
     *
     * @param[out] rParams  Параметры ExtAsk.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetExtAskParams(ilg_extask_params& rParams);

    /**
     * @brief Синхронизирует часы контроллера с часами ПК.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Получить текущее время контроллера можно с помощью функции @ref ReadRtcParams.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SyncClock();

    /**
     * @brief Запускает асинхронную команду синхронизации часов контроллера с часами ПК.
     *
     * @remark Получить текущее время контроллера можно с помощью функции @ref ReadRtcParams.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SyncClock();

    /**
     * @brief Вкл/выкл старый режим аварийного открывания дверей.
     *
     * @param[in] fEnable  `true`, включить аварийный режим.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetEmergencyMode(bool fEnable);

    /**
     * @brief Запускает асинхронную команду вкл/выкл старого режима аварийного открывания дверей.
     *
     * @param[in] fEnable  `true`, включить аварийный режим.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetEmergencyMode(bool fEnable);

    /**
     * @brief Читает параметры AntiCovid.
     *
     * @param[out] pMax8    Максимумы, массив из 8 элементов.
     * @param[out] pCount8  Счётчики, массив из 8 элементов.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadAntiCovid(uint16_t* pMax8, uint16_t* pCount8);

    /**
     * @brief Запускает асинхронную команду чтения параметров AntiCovid.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadAntiCovid();

    /**
     * @brief Возвращает результат чтения параметров AntiCovid.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadAntiCovid.
     * @param[out] pMax8     Максимумы, массив из 8 элементов.
     * @param[out] pCount8   Счётчики, массив из 8 элементов.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadAntiCovid(ilg_handle hCommand, uint16_t* pMax8, uint16_t* pCount8);

    /**
     * @brief Пишет максимумы AntiCovid.
     *
     * @param[in] pMax8  Максимумы, массив из 8 элементов.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteAntiCovid(const uint16_t* pMax8);

    /**
     * @brief Запускает асинхронную команду записи максимумов AntiCovid.
     *
     * @param[in] pMax8  Максимумы, массив из 8 элементов.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteAntiCovid(const uint16_t* pMax8);

    /**
     * @brief Открывает дверь.
     *
     * @param[in] fOut  `true`, открывает для выхода, иначе - для входа.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void OpenDoor(bool fOut);

    /**
     * @brief Запускает асинхронную команду открытия двери.
     *
     * @param[in] fOut  `true`, открывает для выхода, иначе - для входа.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_OpenDoor(bool fOut);

    /**
     * @brief Вкл/выкл противопожарный режим.
     *
     * @param[in] fOn  `true`, включить режим.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetFireMode(bool fOn);

    /**
     * @brief Запускает асинхронную команду вкл/выкл противопожарного режима.
     *
     * @param[in] fOn  `true`, включить режим.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetFireMode(bool fOn);

    /**
     * @brief Вкл/выкл режим охраны.
     *
     * @param[in] fOn  `true`, включить режим.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetSecurityMode(bool fOn);

    /**
     * @brief Запускает асинхронную команду вкл/выкл режима охраны.
     *
     * @param[in] fOn  `true`, включить режим.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetSecurityMode(bool fOn);

    /**
     * @brief Вкл/выкл режим тревоги.
     *
     * @param[in] fOn  `true`, включить режим.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetAlarmMode(bool fOn);

    /**
     * @brief Запускает асинхронную команду вкл/выкл режима тревоги.
     *
     * @param[in] fOn  `true`, включить режим.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetAlarmMode(bool fOn);

    /**
     * @brief Вкл/выкл питание в режиме Электроконтроль.
     *
     * @param[in] fOn  `true`, включить питание.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetEcPower(bool fOn);

    /**
     * @brief Запускает асинхронную команду вкл/выкл питания в режиме Электроконтроль.
     *
     * @param[in] fOn  `true`, включить питание.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetEcPower(bool fOn);

    /**
     * @brief Читает параметры противопожарного режима.
     *
     * @param[out] nSrcMask     Маска разрешения/запрещения источников.
     *  Бит 0 - разрешен пожарный режим по входу FIRE.
     *  Бит 1 - разрешен пожарный режим по превышению температуры.
     * @param[out] nLimitT      Пороговая температура.
     * @param[out] nState       Флаги состояния.
     *  Бит 0 - пожарный режим включён.
     *  Бит 1 - активен пожарный режим по входу FIRE.
     *  Бит 2 - активен пожарный режим по превышению температуры.
     *  Бит 3 - активен пожарный режим по внешней команде.
     * @param[out] nT           Текущая температура.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadFireParams(uint8_t& nSrcMask, uint8_t& nLimitT, uint8_t& nState, uint8_t& nT);

    /**
     * @brief Запускает асинхронную команду чтения параметров противопожарного режима.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadFireParams();

    /**
     * @brief Возвращает результат чтения параметров противопожарного режима.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadFireParams.
     * @param[out] nSrcMask  Маска разрешения/запрещения источников.
     * @param[out] nLimitT   Пороговая температура.
     * @param[out] nState    Флаги состояния.
     * @param[out] nT        Текущая температура.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadFireParams(ilg_handle hCommand, uint8_t& nSrcMask, uint8_t& nLimitT,
                                          uint8_t& nState, uint8_t& nT);

    /**
     * @brief Пишет параметры противопожарного режима.
     *
     * @param[in]  nSrcMask  Маска разрешения/запрещения источников.
     * @param[in]  nLimitT   Пороговая температура.
     * @param[out] pState    Флаги состояния.
     * @param[out] pT        Текущая температура.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteFireParams(uint8_t nSrcMask, uint8_t nLimitT, uint8_t* pState = nullptr,
                                uint8_t* pT = nullptr);

    /**
     * @brief Запускает асинхронную команду записи параметров противопожарного режима.
     *
     * @param[in] nSrcMask  Маска разрешения/запрещения источников.
     * @param[in] nLimitT   Пороговая температура.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteFireParams(uint8_t nSrcMask, uint8_t nLimitT);

    /**
     * @brief Возвращает результат записи параметров противопожарного режима.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_WriteFireParams.
     * @param[out] nState    Флаги состояния.
     * @param[out] nT        Текущая температура.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_WriteFireParams(ilg_handle hCommand, uint8_t& nState, uint8_t& nT);

    /**
     * @brief Читает параметры режима охраны.
     *
     * @param[out] nSrcMask  Маска разрешения/запрещения источников.
     *  Бит 0 - разрешена тревога по входу ALARM.
     *  Бит 1 - разрешена тревога по тамперу.
     *  Бит 2 - разрешена тревога по датчику двери.
     * @param[out] nTime     Время звучания сирены после исчезновения источника тревоги.
     * @param[out] nState    Флаги состояния.
     *  Бит 0 - охранный режим включён.
     *  Бит 1 - тревога включена.
     *  Бит 2 - тревога по входу ALARM.
     *  Бит 3 - тревога по тамперу.
     *  Бит 4 - тревога по датчику двери.
     *  Бит 5 - тревога включена по сети.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadSecurityParams(uint8_t& nSrcMask, uint8_t& nTime, uint8_t& nState);

    /**
     * @brief Запускает асинхронную команду чтения параметров режима охраны.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadSecurityParams();

    /**
     * @brief Возвращает результат чтения параметров режима охраны.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadSecurityParams.
     * @param[out] nSrcMask  Маска разрешения/запрещения источников.
     * @param[out] nTime     Время звучания сирены после исчезновения источника тревоги.
     * @param[out] nState    Флаги состояния.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadSecurityParams(ilg_handle hCommand, uint8_t& nSrcMask,
                                              uint8_t& nTime, uint8_t& nState);

    /**
     * @brief Пишет параметры режима охраны.
     *
     * @param[in]  nSrcMask  Маска разрешения/запрещения источников.
     * @param[in]  nTime     Время звучания сирены после исчезновения источника тревоги.
     * @param[out] pState    Флаги состояния.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteSecurityParams(uint8_t nSrcMask, uint8_t nTime, uint8_t* pState = nullptr);

    /**
     * @brief Запускает асинхронную команду записи параметров режима охраны.
     *
     * @param[in] nSrcMask  Маска разрешения/запрещения источников.
     * @param[in] nTime     Время звучания сирены после исчезновения источника тревоги.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteSecurityParams(uint8_t nSrcMask, uint8_t nTime);

    /**
     * @brief Возвращает результат записи параметров режима охраны.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_WriteSecurityParams.
     * @param[out] nState    Флаги состояния.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_WriteSecurityParams(ilg_handle hCommand, uint8_t& nState);

    /**
     * @brief Читает параметры режима ЭлектроКонтроль.
     *
     * @param[out] nPowerConfig  Флаги конфигурации режима.
     *  Бит 0 (ILG_ECPC_F_ENABLED) Управление питанием включено.
     *  Бит 1 (ILG_ECPC_F_SCHEDULE) Использовать временную зону 6 для включения питания.
     *  Бит 2 (ILG_ECPC_F_EXT_READER) Контрольный считыватель: «0» Matrix-II Net, «1» внешний
     * считыватель. Бит 3 (ILG_ECPC_F_INVERT) Инвертировать управляющий выход.
     *  Бит 3 (ILG_ECPC_F_INVERT) Инвертировать управляющий выход.
     *  Бит 4 (ILG_ECPC_F_EXIT_OFF) Задействовать датчик двери.
     *  Бит 5 (ILG_ECPC_F_CARD_OPEN) Не блокировать функцию открывания для контрольного считывателя.
     * @param[out] nDelay        Время задержки в секундах.
     * @param[out] pState        Флаги состояния.
     *  Бит 0 - питание включено.
     *  Бит 1 - активно включение по временной зоне.
     *  Бит 2 - включено командой по сети
     *  Бит 3 - идет отработка задержки.
     *  Бит 4 - карта в поле контрольного считывателя.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadEcParams(uint8_t& nPowerConfig, uint8_t& nDelay, uint8_t* pState = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения параметров режима ЭлектроКонтроль.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadEcParams();

    /**
     * @brief Возвращает результат чтения параметров режима ЭлектроКонтроль.
     *
     * @param[in]  hCommand     Дескриптор команды, который вернула @ref Begin_ReadEcParams.
     * @param[out] nPowerConfig Флаги конфигурации режима.
     * @param[out] nDelay       Время задержки в секундах.
     * @param[out] pState       Флаги состояния.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadEcParams(ilg_handle hCommand, uint8_t& nPowerConfig, uint8_t& nDelay,
                                        uint8_t* pState = nullptr);

    /**
     * @brief Пишет режим контроллера.
     *
     * Устанавливает в контроллер текущий режим контроллера.
     *
     * @param[in] nMode  Режим контроллера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteMode(ilg_controller_mode nMode);

    /**
     * @brief Запускает асинхронную команду записи режима контроллера.
     *
     * @param[in] nMode  Режим контроллера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteMode(ilg_controller_mode nMode);

    /**
     * @brief Читает режим контроллера.
     *
     * Возвращает текущий режим контроллера.
     *
     * @param[out] pFlags  Флаги активизации.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @return Режим контроллера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline ilg_controller_mode ReadMode(uint8_t* pFlags = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения режима контроллера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadMode();

    /**
     * @brief Возвращает результат чтения режима контроллера.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadMode.
     * @param[out] nMode     Режим контроллера.
     * @param[out] pFlags    Флаги активизации.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadMode(ilg_handle hCommand, ilg_controller_mode& nMode,
                                    uint8_t* pFlags = nullptr);

    /**
     * @brief Пишет конфигурацию контроллера.
     *
     * @param[in] pData  Данные конфигурации.
     * @param[in] nSize  Размер данных в байтах.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Данные конфигурации можно получить с помощью ПО GuardLight.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteConfiguration(const void* pData, size_t nSize);

    /**
     * @brief Запускает асинхронную команду записи конфигурации контроллера.
     *
     * @param[in] pData  Данные конфигурации.
     * @param[in] nSize  Размер данных в байтах.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteConfiguration(const void* pData, size_t nSize);

    /**
     * @brief Читает конфигурацию контроллера.
     *
     * @param[out] oData  Данные конфигурации.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    void ReadConfiguration(std::vector<uint8_t>& oData);

    /**
     * @brief Запускает асинхронную команду чтения конфигурации контроллера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadConfiguration();

    /**
     * @brief Возвращает результат чтения конфигурации контроллера.
     *
     * @param[in]  hCommand  Дескриптор команды, который вернула @ref Begin_ReadConfiguration.
     * @param[out] oData     Данные конфигурации.
     *
     * @exception CILGException если произошла ошибка.
     */
    static void End_ReadConfiguration(ilg_handle hCommand, std::vector<uint8_t>& oData);

    /**
     * @brief Устанавливает сетевой адрес контроллеру.
     *
     * @param[in] nAddress  Адрес контроллера 0..254.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Работает только в режиме Normal конвертера или с Z-397 Guard.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetNetworkAddress(uint8_t nAddress);

    /**
     * @brief Запускает асинхронную команду установки сетевого адреса.
     *
     * @param[in] nAddress  Адрес контроллера 0..254.
     *
     * @exception CILGException если произошла ошибка.
     *
     * @return Класс дескриптора команды.
     */
    inline CAsyncCommand Begin_SetNetworkAddress(uint8_t nAddress);

    /**
     * @brief Устанавливает прошивку контроллера.
     *
     * @param[in] pData  Данные прошивки.
     * @param[in] nSize  Размер данных прошивки в байтах.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetFirmware(const void* pData, size_t nSize);

    /**
     * @brief Запускает асинхронную команду установки прошивки контроллера.
     *
     * @param[in] pData  Данные прошивки.
     * @param[in] nSize  Размер данных прошивки в байтах.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetFirmware(const void* pData, size_t nSize);

    /**
     * @brief Определяет тип события контроллера и формат записи события.
     *
     * @param[in]  nEvent   Данные события.
     * @param[out] pFormat  Формат записи события.
     *
     * @return Тип события.
     * @exception CILGException если произошла ошибка.
     */
    inline ilg_event_type DecodeEventType(const uint64_t& nEvent,
                                          ilg_event_format* pFormat = nullptr);

    /**
     * @brief Декодирует событие прохода.
     *
     * @param[in]  nEvent      Данные события.
     * @param[out] rTime       Дата и время события.
     * @param[out] nDirection  Направление прохода.
     * @param[out] nKeyBankN   Номер банка ключей.
     * @param[out] nKeyIdx     Позиция ключа в банке ключей.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void DecodePassageEvent(const uint64_t& nEvent, ilg_controller_time& rTime,
                                   ilg_direction& nDirection, uint8_t& nKeyBankN, ssize_t& nKeyIdx);

    /**
     * @brief Декодирует событие с датой и временем.
     *
     * @param[in]  nEvent  Данные события.
     * @param[out] rTime   Дата и время события.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void DecodeTimeEvent(const uint64_t& nEvent, ilg_controller_time& rTime);

    /**
     * @brief Декодирует событие переключения режима контроллера.
     *
     * @param[in]  nEvent    Данные события.
     * @param[out] rTime     Дата и время события.
     * @param[out] nMode     Режим контроллера.
     * @param[out] nFlags    Флаги состояния.
     * @param[out] nTrigger  Код условия вызвавшего срабатывание.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void DecodeControllerModeEvent(const uint64_t& nEvent, ilg_controller_time& rTime,
                                          ilg_controller_mode& nMode, uint8_t& nFlags,
                                          uint8_t& nTrigger);

    /**
     * @brief Декодирует событие изменения состояния.
     *
     * @param[in]  nEvent    Данные события.
     * @param[out] rTime     Дата и время события.
     * @param[out] nFlags    Флаги состояния.
     * @param[out] nTrigger  Код условия вызвавшего срабатывание.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void DecodeStateEvent(const uint64_t& nEvent, ilg_controller_time& rTime,
                                 uint8_t& nFlags, uint8_t& nTrigger);

    /**
     * @brief Декодирует событие с номером ключа.
     *
     * @param[in]  nEvent      Данные события.
     * @param[out] rKeyNumber  Номер ключа.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void DecodeKeyNumber(const uint64_t& nEvent, ilg_key_number& rKeyNumber);

    /**
     * @brief Клонирует дескриптор контроллера.
     * @return Дескриптор контроллера.
     */
    inline CController Clone() const;
};

inline void CController::SetMessageCallback(ilg_controller_message_callback pCallback,
                                            void* pUserData) {
    ILGCheck(ilg_controller_set_message_callback(m_h, pCallback, pUserData));
}

inline void CController::EnableMessageQueue(bool fEnable) {
    ILGCheck(ilg_controller_enable_message_queue(m_h, static_cast<ilg_bool>(fEnable)));
}

inline bool CController::GetMessage(ilg_controller_msg& nMsg, const void*& pMsgData) const {
    ilg_bool res;
    ILGCheck(ilg_controller_get_message(m_h, &nMsg, &pMsgData, &res));
    return static_cast<bool>(res);
}

inline void CController::SetOptions(const ilg_controller_options& rOptions) {
    ILGCheck(ilg_controller_set_options(m_h, &rOptions));
}

inline void CController::GetOptions(ilg_controller_options& rOptions) {
    ILGCheck(ilg_controller_get_options(m_h, &rOptions));
}

inline void CController::Connect(bool fReconnect) {
    ILGCheck(ilg_controller_connect(m_h, static_cast<ilg_bool>(fReconnect)));
}

inline CAsyncCommand CController::Begin_Connect(bool fReconnect) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_connect(m_h, static_cast<ilg_bool>(fReconnect), &res.m_h));
    return res;
}

inline void CController::Disconnect() {
    ILGCheck(ilg_controller_disconnect(m_h));
}

inline CAsyncCommand CController::Begin_Disconnect() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_disconnect(m_h, &res.m_h));
    return res;
}

inline ilg_connection_status CController::GetConnectionStatus() const {
    ilg_connection_status res;
    ILGCheck(ilg_controller_get_connection_status(m_h, &res));
    return res;
}

inline void CController::GetControllerInfo(ilg_controller_info& rInfo) const {
    ILGCheck(ilg_controller_get_info(m_h, &rInfo));
}

inline void CController::GetConverterInfo(ilg_converter_info& rInfo) const {
    ILGCheck(ilg_controller_get_converter_info(m_h, &rInfo));
}

inline CAsyncCommand CController::Begin_ReadLines() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_lines(m_h, &res.m_h));
    return res;
}

inline void CController::ReadLockTimes(uint8_t nBankN, uint16_t& nOpen, uint16_t& nLet,
                                       uint16_t& nMax, bool& fBigTime) {
    ilg_bool f;
    ILGCheck(ilg_controller_read_lock_times(m_h, nBankN, &nOpen, &nLet, &nMax, &f));
    fBigTime = f;
}

inline CAsyncCommand CController::Begin_ReadLockTimes(uint8_t nBankN) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_lock_times(m_h, nBankN, &res.m_h));
    return res;
}

inline void CController::End_ReadLockTimes(ilg_handle hCommand, uint16_t& nOpen, uint16_t& nLet,
                                           uint16_t& nMax, bool& fBigTime) {
    ilg_bool f;
    ILGCheck(ilg_controller_end_read_lock_times(hCommand, &nOpen, &nLet, &nMax, &f));
    fBigTime = f;
}

inline void CController::WriteLockTimes(uint8_t nBankN, uint16_t nOpen, uint16_t nLet,
                                        uint16_t nMax, bool fBigTime) {
    ILGCheck(ilg_controller_write_lock_times(m_h, nBankN, nOpen, nLet, nMax,
                                             static_cast<ilg_bool>(fBigTime)));
}

inline CAsyncCommand CController::Begin_WriteLockTimes(uint8_t nBankN, uint16_t nOpen,
                                                       uint16_t nLet, uint16_t nMax,
                                                       bool fBigTime) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_lock_times(m_h, nBankN, nOpen, nLet, nMax,
                                                   static_cast<ilg_bool>(fBigTime), &res.m_h));
    return res;
}

inline void CController::ReadApbTime(bool& fEnabled, uint16_t& nMinutes) {
    ilg_bool f;
    ILGCheck(ilg_controller_read_apb_time(m_h, &f, &nMinutes));
    fEnabled = static_cast<bool>(f);
}

inline CAsyncCommand CController::Begin_ReadApbTime() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_apb_time(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadApbTime(ilg_handle hCommand, bool& fEnabled, uint16_t& nMinutes) {
    ilg_bool f;
    ILGCheck(ilg_controller_end_read_apb_time(hCommand, &f, &nMinutes));
    fEnabled = static_cast<bool>(f);
}

inline void CController::WriteApbTime(bool fEnabled, uint16_t nMinutes) {
    ILGCheck(ilg_controller_write_apb_time(m_h, static_cast<ilg_bool>(fEnabled), nMinutes));
}

inline CAsyncCommand CController::Begin_WriteApbTime(bool fEnabled, uint16_t nMinutes) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_apb_time(m_h, static_cast<ilg_bool>(fEnabled), nMinutes,
                                                 &res.m_h));
    return res;
}

inline void CController::ReadTimeZones(uint8_t nBankN, size_t nIdx, ilg_time_zone* pBuf,
                                       size_t nCount, size_t* pRead) {
    ILGCheck(ilg_controller_read_time_zones(m_h, nBankN, nIdx, pBuf, nCount, pRead));
}

inline CAsyncCommand CController::Begin_ReadTimeZones(uint8_t nBankN, size_t nIdx, size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_time_zones(m_h, nBankN, nIdx, nCount, &res.m_h));
    return res;
}

inline void CController::End_ReadTimeZones(ilg_handle hCommand, const ilg_time_zone*& pList,
                                           size_t& nRead) {
    ILGCheck(ilg_controller_end_read_time_zones(hCommand, &pList, &nRead));
}

inline void CController::WriteTimeZones(uint8_t nBankN, size_t nIdx, const ilg_time_zone* pTZs,
                                        size_t nCount, size_t* pWritten) {
    ILGCheck(ilg_controller_write_time_zones(m_h, nBankN, nIdx, pTZs, nCount, pWritten));
}

inline CAsyncCommand CController::Begin_WriteTimeZones(uint8_t nBankN, size_t nIdx,
                                                       const ilg_time_zone* pTZs, size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_time_zones(m_h, nBankN, nIdx, pTZs, nCount, &res.m_h));
    return res;
}

inline void CController::End_WriteTimeZones(ilg_handle hCommand, size_t& nWritten) {
    ILGCheck(ilg_controller_end_write_time_zones(hCommand, &nWritten));
}

inline void CController::ReadModeTimeZones(size_t nIdx, ilg_mode_time_zone* pBuf, size_t nCount) {
    ILGCheck(ilg_controller_read_mode_time_zones(m_h, nIdx, pBuf, nCount));
}

inline CAsyncCommand CController::Begin_ReadModeTimZones(size_t nIdx, size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_mode_time_zones(m_h, nIdx, nCount, &res.m_h));
    return res;
}

inline void CController::End_ReadModeTimZones(ilg_handle hCommand, const ilg_mode_time_zone*& pList,
                                              size_t& nRead) {
    ILGCheck(ilg_controller_end_read_mode_time_zones(hCommand, &pList, &nRead));
}

inline void CController::WriteModeTimeZones(size_t nIdx, const ilg_mode_time_zone* pTZs,
                                            size_t nCount) {
    ILGCheck(ilg_controller_write_mode_time_zones(m_h, nIdx, pTZs, nCount));
}

inline CAsyncCommand CController::Begin_WriteModeTimeZones(size_t nIdx,
                                                           const ilg_mode_time_zone* pTZs,
                                                           size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_mode_time_zones(m_h, nIdx, pTZs, nCount, &res.m_h));
    return res;
}

inline void CController::ReadKeys(uint8_t nBankN, size_t nIdx, ilg_key* pBuf, size_t nCount,
                                  size_t* pRead) {
    ILGCheck(ilg_controller_read_keys(m_h, nBankN, nIdx, pBuf, nCount, pRead));
}

inline CAsyncCommand CController::Begin_ReadKeys(uint8_t nBankN, size_t nIdx, size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_keys(m_h, nBankN, nIdx, nCount, &res.m_h));
    return res;
}

inline void CController::End_ReadKeys(ilg_handle hCommand, const ilg_key*& pList, size_t& nRead) {
    ILGCheck(ilg_controller_end_read_keys(hCommand, &pList, &nRead));
}

inline void CController::WriteKeys(uint8_t nBankN, size_t nIdx, const ilg_key* pKeys, size_t nCount,
                                   size_t* pWritten, bool fSelected) {
    ILGCheck(ilg_controller_write_keys(m_h, nBankN, nIdx, pKeys, nCount, pWritten,
                                       static_cast<ilg_bool>(fSelected)));
}

inline CAsyncCommand CController::Begin_WriteKeys(uint8_t nBankN, size_t nIdx, const ilg_key* pKeys,
                                                  size_t nCount, bool fSelected) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_keys(m_h, nBankN, nIdx, pKeys, nCount,
                                             static_cast<ilg_bool>(fSelected), &res.m_h));
    return res;
}

inline void CController::End_WriteKeys(ilg_handle hCommand, size_t& nWritten) {
    ILGCheck(ilg_controller_end_write_keys(hCommand, &nWritten));
}

inline void CController::EraseKeys(uint8_t nBankN, size_t nIdx, size_t nCount, size_t* pErased) {
    ILGCheck(ilg_controller_erase_keys(m_h, nBankN, nIdx, nCount, pErased));
}

inline CAsyncCommand CController::Begin_EraseKeys(uint8_t nBankN, size_t nIdx, size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_erase_keys(m_h, nBankN, nIdx, nCount, &res.m_h));
    return res;
}

inline void CController::End_EraseKeys(ilg_handle hCommand, size_t& nErased) {
    ILGCheck(ilg_controller_end_erase_keys(hCommand, &nErased));
}

inline void CController::EraseKeys2(uint8_t nBankN, const size_t* pIdxs, size_t nCount,
                                    size_t* pErased) {
    ILGCheck(ilg_controller_erase_keys2(m_h, nBankN, pIdxs, nCount, pErased));
}

inline CAsyncCommand CController::Begin_EraseKeys2(uint8_t nBankN, const size_t* pIdxs,
                                                   size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_erase_keys2(m_h, nBankN, pIdxs, nCount, &res.m_h));
    return res;
}

inline void CController::End_EraseKeys2(ilg_handle hCommand, size_t& nErased) {
    ILGCheck(ilg_controller_end_erase_keys2(hCommand, &nErased));
}

inline ssize_t CController::ReadKeyTopIdx(uint8_t nBankN) {
    ssize_t res;
    ILGCheck(ilg_controller_read_key_top_idx(m_h, nBankN, &res));
    return res;
}

inline CAsyncCommand CController::Begin_ReadKeyTopIdx(uint8_t nBankN) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_key_top_idx(m_h, nBankN, &res.m_h));
    return res;
}

inline void CController::End_ReadKeyTopIdx(ilg_handle hCommand, ssize_t& nIdx) {
    ILGCheck(ilg_controller_end_read_key_top_idx(hCommand, &nIdx));
}

inline void CController::ReadEvents(size_t nIdx, uint64_t* pBuf, size_t nCount, size_t* pRead) {
    ILGCheck(ilg_controller_read_events(m_h, nIdx, pBuf, nCount, pRead));
}

inline CAsyncCommand CController::Begin_ReadEvents(size_t nIdx, size_t nCount) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_events(m_h, nIdx, nCount, &res.m_h));
    return res;
}

inline void CController::End_ReadEvents(ilg_handle hCommand, const uint64_t*& pList,
                                        size_t& nRead) {
    ILGCheck(ilg_controller_end_read_events(hCommand, &pList, &nRead));
}

inline void CController::WriteEventReadIdx(size_t nIdx) {
    ILGCheck(ilg_controller_write_event_read_idx(m_h, nIdx));
}

inline CAsyncCommand CController::Begin_WriteEventReadIdx(size_t nIdx) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_event_read_idx(m_h, nIdx, &res.m_h));
    return res;
}

inline void CController::ReadRtcParams(ilg_rtc_params& rParams) {
    ILGCheck(ilg_controller_read_rtc_params(m_h, &rParams));
}

inline CAsyncCommand CController::Begin_ReadRtcParams() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_rtc_params(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadRtcParams(ilg_handle hCommand, ilg_rtc_params& rParams) {
    ILGCheck(ilg_controller_end_read_rtc_params(hCommand, &rParams));
}

inline void CController::GetRtcParams(ilg_rtc_params& rParams) {
    ILGCheck(ilg_controller_get_rtc_params(m_h, &rParams));
}

inline void CController::ReadExtAskParams(ilg_extask_params& rParams) {
    ILGCheck(ilg_controller_read_extask_params(m_h, &rParams));
}

inline CAsyncCommand CController::Begin_ReadExtAskParams() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_extask_params(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadExtAskParams(ilg_handle hCommand, ilg_extask_params& rParams) {
    ILGCheck(ilg_controller_end_read_extask_params(hCommand, &rParams));
}

inline void CController::GetExtAskParams(ilg_extask_params& rParams) {
    ILGCheck(ilg_controller_get_extask_params(m_h, &rParams));
}

inline void CController::SyncClock() {
    ILGCheck(ilg_controller_sync_clock(m_h));
}

inline CAsyncCommand CController::Begin_SyncClock() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_sync_clock(m_h, &res.m_h));
    return res;
}

inline void CController::SetEmergencyMode(bool fEnable) {
    ILGCheck(ilg_controller_set_emergency_mode(m_h, static_cast<ilg_bool>(fEnable)));
}

inline CAsyncCommand CController::Begin_SetEmergencyMode(bool fEnable) {
    CAsyncCommand res;
    ILGCheck(
        ilg_controller_begin_set_emergency_mode(m_h, static_cast<ilg_bool>(fEnable), &res.m_h));
    return res;
}

inline void CController::ReadAntiCovid(uint16_t* pMax8, uint16_t* pCount8) {
    ILGCheck(ilg_controller_read_anticovid(m_h, pMax8, pCount8));
}

inline CAsyncCommand CController::Begin_ReadAntiCovid() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_anticovid(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadAntiCovid(ilg_handle hCommand, uint16_t* pMax8,
                                           uint16_t* pCount8) {
    ILGCheck(ilg_controller_end_read_anticovid(hCommand, pMax8, pCount8));
}

inline void CController::WriteAntiCovid(const uint16_t* pMax8) {
    ILGCheck(ilg_controller_write_anticovid(m_h, pMax8));
}

inline CAsyncCommand CController::Begin_WriteAntiCovid(const uint16_t* pMax8) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_anticovid(m_h, pMax8, &res.m_h));
    return res;
}

inline void CController::OpenDoor(bool fOut) {
    ILGCheck(ilg_controller_open_door(m_h, static_cast<ilg_bool>(fOut)));
}

inline CAsyncCommand CController::Begin_OpenDoor(bool fOut) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_open_door(m_h, static_cast<ilg_bool>(fOut), &res.m_h));
    return res;
}

inline void CController::SetFireMode(bool fOn) {
    ILGCheck(ilg_controller_set_fire_mode(m_h, static_cast<ilg_bool>(fOn)));
}

inline CAsyncCommand CController::Begin_SetFireMode(bool fOn) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_set_fire_mode(m_h, static_cast<ilg_bool>(fOn), &res.m_h));
    return res;
}

inline void CController::SetSecurityMode(bool fOn) {
    ILGCheck(ilg_controller_set_security_mode(m_h, static_cast<ilg_bool>(fOn)));
}

inline CAsyncCommand CController::Begin_SetSecurityMode(bool fOn) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_set_security_mode(m_h, static_cast<ilg_bool>(fOn), &res.m_h));
    return res;
}

inline void CController::SetAlarmMode(bool fOn) {
    ILGCheck(ilg_controller_set_alarm_mode(m_h, static_cast<ilg_bool>(fOn)));
}

inline CAsyncCommand CController::Begin_SetAlarmMode(bool fOn) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_set_alarm_mode(m_h, static_cast<ilg_bool>(fOn), &res.m_h));
    return res;
}

inline void CController::SetEcPower(bool fOn) {
    ILGCheck(ilg_controller_set_ec_power(m_h, static_cast<ilg_bool>(fOn)));
}

inline CAsyncCommand CController::Begin_SetEcPower(bool fOn) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_set_ec_power(m_h, static_cast<ilg_bool>(fOn), &res.m_h));
    return res;
}

inline void CController::ReadFireParams(uint8_t& nSrcMask, uint8_t& nLimitT, uint8_t& nState,
                                        uint8_t& nT) {
    ILGCheck(ilg_controller_read_fire_params(m_h, &nSrcMask, &nLimitT, &nState, &nT));
}

inline CAsyncCommand CController::Begin_ReadFireParams() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_fire_params(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadFireParams(ilg_handle hCommand, uint8_t& nSrcMask,
                                            uint8_t& nLimitT, uint8_t& nState, uint8_t& nT) {
    ILGCheck(ilg_controller_end_read_fire_params(hCommand, &nSrcMask, &nLimitT, &nState, &nT));
}

inline void CController::WriteFireParams(uint8_t nSrcMask, uint8_t nLimitT, uint8_t* pState,
                                         uint8_t* pT) {
    ILGCheck(ilg_controller_write_fire_params(m_h, nSrcMask, nLimitT, pState, pT));
}

inline CAsyncCommand CController::Begin_WriteFireParams(uint8_t nSrcMask, uint8_t nLimitT) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_fire_params(m_h, nSrcMask, nLimitT, &res.m_h));
    return res;
}

inline void CController::End_WriteFireParams(ilg_handle hCommand, uint8_t& nState, uint8_t& nT) {
    ILGCheck(ilg_controller_end_write_fire_params(hCommand, &nState, &nT));
}

inline void CController::ReadSecurityParams(uint8_t& nSrcMask, uint8_t& nTime, uint8_t& nState) {
    ILGCheck(ilg_controller_read_security_params(m_h, &nSrcMask, &nTime, &nState));
}

inline CAsyncCommand CController::Begin_ReadSecurityParams() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_security_params(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadSecurityParams(ilg_handle hCommand, uint8_t& nSrcMask,
                                                uint8_t& nTime, uint8_t& nState) {
    ILGCheck(ilg_controller_end_read_security_params(hCommand, &nSrcMask, &nTime, &nState));
}

inline void CController::WriteSecurityParams(uint8_t nSrcMask, uint8_t nTime, uint8_t* pState) {
    ILGCheck(ilg_controller_write_security_params(m_h, nSrcMask, nTime, pState));
}

inline CAsyncCommand CController::Begin_WriteSecurityParams(uint8_t nSrcMask, uint8_t nTime) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_security_params(m_h, nSrcMask, nTime, &res.m_h));
    return res;
}

inline void CController::End_WriteSecurityParams(ilg_handle hCommand, uint8_t& nState) {
    ILGCheck(ilg_controller_end_write_security_params(hCommand, &nState));
}

inline void CController::ReadEcParams(uint8_t& nPowerConfig, uint8_t& nDelay, uint8_t* pState) {
    ILGCheck(ilg_controller_read_ec_params(m_h, &nPowerConfig, &nDelay, pState));
}

inline CAsyncCommand CController::Begin_ReadEcParams() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_ec_params(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadEcParams(ilg_handle hCommand, uint8_t& nPowerConfig,
                                          uint8_t& nDelay, uint8_t* pState) {
    ILGCheck(ilg_controller_end_read_ec_params(hCommand, &nPowerConfig, &nDelay, pState));
}

inline void CController::WriteMode(ilg_controller_mode nMode) {
    ILGCheck(ilg_controller_write_mode(m_h, nMode));
}

inline CAsyncCommand CController::Begin_WriteMode(ilg_controller_mode nMode) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_mode(m_h, nMode, &res.m_h));
    return res;
}

inline ilg_controller_mode CController::ReadMode(uint8_t* pFlags) {
    ilg_controller_mode res;
    ILGCheck(ilg_controller_read_mode(m_h, &res, pFlags));
    return res;
}

inline CAsyncCommand CController::Begin_ReadMode() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_mode(m_h, &res.m_h));
    return res;
}

inline void CController::End_ReadMode(ilg_handle hCommand, ilg_controller_mode& nMode,
                                      uint8_t* pFlags) {
    ILGCheck(ilg_controller_end_read_mode(hCommand, &nMode, pFlags));
}

inline void CController::WriteConfiguration(const void* pData, size_t nSize) {
    ILGCheck(ilg_controller_write_configuration(m_h, pData, nSize));
}

inline CAsyncCommand CController::Begin_WriteConfiguration(const void* pData, size_t nSize) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_write_configuration(m_h, pData, nSize, &res.m_h));
    return res;
}

inline CAsyncCommand CController::Begin_ReadConfiguration() {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_read_configuration(m_h, &res.m_h));
    return res;
}

inline void CController::SetNetworkAddress(uint8_t nAddress) {
    ILGCheck(ilg_controller_set_network_address(m_h, nAddress));
}

inline CAsyncCommand CController::Begin_SetNetworkAddress(uint8_t nAddress) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_set_network_address(m_h, nAddress, &res.m_h));
    return res;
}

inline void CController::SetFirmware(const void* pData, size_t nSize) {
    ILGCheck(ilg_controller_set_firmware(m_h, pData, nSize));
}

inline CAsyncCommand CController::Begin_SetFirmware(const void* pData, size_t nSize) {
    CAsyncCommand res;
    ILGCheck(ilg_controller_begin_set_firmware(m_h, pData, nSize, &res.m_h));
    return res;
}

inline ilg_event_type CController::DecodeEventType(const uint64_t& nEvent,
                                                   ilg_event_format* pFormat) {
    ilg_event_type res;
    ILGCheck(ilg_controller_decode_event_type(m_h, nEvent, pFormat, &res));
    return res;
}

inline void CController::DecodePassageEvent(const uint64_t& nEvent, ilg_controller_time& rTime,
                                            ilg_direction& nDirection, uint8_t& nKeyBankN,
                                            ssize_t& nKeyIdx) {
    ILGCheck(ilg_controller_decode_passage_event(m_h, nEvent, &rTime, &nDirection, &nKeyBankN,
                                                 &nKeyIdx));
}

inline void CController::DecodeTimeEvent(const uint64_t& nEvent, ilg_controller_time& rTime) {
    ILGCheck(ilg_controller_decode_time_event(m_h, nEvent, &rTime));
}

inline void CController::DecodeControllerModeEvent(const uint64_t& nEvent,
                                                   ilg_controller_time& rTime,
                                                   ilg_controller_mode& nMode, uint8_t& nFlags,
                                                   uint8_t& nTrigger) {
    ILGCheck(ilg_controller_decode_controller_mode_event(m_h, nEvent, &rTime, &nMode, &nFlags,
                                                         &nTrigger));
}

inline void CController::DecodeStateEvent(const uint64_t& nEvent, ilg_controller_time& rTime,
                                          uint8_t& nFlags, uint8_t& nTrigger) {
    ILGCheck(ilg_controller_decode_state_event(m_h, nEvent, &rTime, &nFlags, &nTrigger));
}

inline void CController::DecodeKeyNumber(const uint64_t& nEvent, ilg_key_number& rKeyNumber) {
    ILGCheck(ilg_controller_decode_key_number(m_h, nEvent, &rKeyNumber));
}

inline CController CController::Clone() const {
    ilg_handle hNewHandle;
    ILGCheck(ilg_clone_handle(m_h, &hNewHandle));
    return CController(hNewHandle);
}

/**
 * @brief Класс конвертера.
 *
 * Класс-обёртка для дескриптора конвертера.
 */
class CConverter : public CILGHandle {
public:
    /// Конструктор по умолчанию.
    CConverter();

    /**
     * @brief Конструктор класса из дескриптора конвертера.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    CConverter(ilg_handle h);

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой конвертер.
     */
    CConverter(CConverter&& other);

    /// Деструктор
    virtual ~CConverter();

    /// Оператор перемещения
    CConverter& operator=(CConverter&& other);

    /**
     * @brief Устанавливает функцию обратного вызова для уведомлений конвертера.
     *
     * Устанавливает функцию для получения сообщений от дескриптора конвертера.
     *
     * @param[in] pCallback  Указатель на функцию, которую библиотека будет вызывать при
     * возникновении события конвертера.
     * @param[in] pUserData  Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе вернёт ошибку
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
     */
    inline void SetMessageCallback(ilg_converter_message_callback pCallback,
                                   void* pUserData = nullptr);

    /**
     * @brief Включает/выключает очередь сообщений.
     *
     * Эта функция устанавливает/снимает флаг в дескрипторе конвертера.
     * Очередь сообщений предназначена для синхронизации обработки сообщений.
     *
     * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
     * вызывается функция обратного вызова, установленная функцией @ref
     * SetMessageCallback, из которой посылается сигнал потоку, обрабатывающему
     * сообщения, этот поток при получении сигнала циклично вызывает @ref GetMessage, чтобы получить
     * и обработать все сообщения.
     *
     * @param[in] fEnable  `true`, включает очередь, иначе - выключает.
     *
     * @warning Если не извлекать сообщения из очереди функцией @ref GetMessage, то она будет расти
     * пока не закончится память.
     *
     * @warning Если не извлекать сообщения из очереди функцией @ref GetMessage, то она будет расти
     * пока не закончится память.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline void EnableMessageQueue(bool fEnable = true);

    /**
     * @brief Извлекает следующее сообщение из очереди.
     *
     * @param[out] nMsg      Тип сообщения.
     * @param[out] pMsgData  Указатель на данные пользователя, которые были установлены функцией
     * @ref SetMessageCallback.
     *
     * @return `true`, если сообщение успешно извлечено, иначе - очередь пуста.
     *
     * @exception CILGException(ILG_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetMessage(ilg_converter_msg& nMsg, const void*& pMsgData) const;

    /**
     * @brief Устанавливает параметры конвертера.
     *
     * @param[in] rOptions   Параметры конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetOptions(const ilg_converter_options& rOptions);

    /**
     * @brief Возвращает параметры конвертера.
     *
     * @param[in] rOptions    Параметры конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetOptions(ilg_converter_options& rOptions) const;

    /**
     * @brief Подключается к конвертеру.
     *
     * @param[in] fReconnect  `true`, Отключается перед подключением.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Если подключиться не удалось и период автоподключения (nReconnectPeriod в @ref GetOptions) не
     * равен -1, то продолжает периодически пытаться подключиться.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void Connect(bool fReconnect = false);

    /**
     * @brief Запускает асинхронную команду подключения к конвертеру.
     *
     * @param[in]  fReconnect  `true`, переподключиться.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_Connect(bool fReconnect = false);

    /**
     * @brief Отключается от конвертера.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Запрещает автоподключение к конвертеру.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void Disconnect();

    /**
     * @brief Запускает асинхронную команду отключения от конвертера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_Disconnect();

    /**
     * @brief Возвращает состояние подключения к конвертеру.
     *
     * @return Состояние подключения к конвертеру.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline ilg_connection_status GetConnectionStatus() const;

    /**
     * @brief Возвращает информацию о конвертере.

     * @param[out] rInfo       Информация о конвертере.
     *
     * @warning В @p rInfo ссылки `pszPortName` и `pszConnect` действительны пока жив этот класс
     дескриптора.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetConverterInfo(ilg_converter_info& rInfo) const;

    /**
     * @brief Возвращает информацию о текущей лицензии конвертера.
     *
     * @param[out] rInfo   Информация о лицензии.
     * @param[in]  fForce  `true`, прочитать инфо из конвертера, иначе взять из кэша.
     *
     * @remark Работает только в режиме `Advanced` конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadLicenseInfo(ilg_license_info& rInfo, bool fForce = false);

    /**
     * @brief Читает информацию о лицензии конвертера.
     *
     * @param[in]  nLicenseN   Номер лицензии, информацию о которой нужно получить.
     * @param[out] rInfo       Информация о лицензии.
     *
     * @remark Работает только в режиме `Advanced` конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ReadLicenseInfo(uint32_t nLicenseN, ilg_license_info& rInfo);

    /**
     * @brief Запускает асинхронную команду запроса инфо о лицензии конвертера.
     *
     * @param[in] nLicenseN  Номер лицензии, информацию о которой нужно получить.
     *
     * @remark Работает только в режиме `Advanced` конвертера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ReadLicenseInfo(uint32_t nLicenseN);

    /**
     * @brief Возвращает результат запроса инфо о лицензии конвертера.
     *
     * @param[in]  hCommand  Дескриптор команды, которую вернула функция @ref Begin_ReadLicenseInfo.
     * @param[out] rInfo     Информация о лицензии конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_ReadLicenseInfo(ilg_handle hCommand, ilg_license_info& rInfo);

    /**
     * @brief Возвращает количество установленных лицензий.
     *
     * @warning Эта функция читает из контроллера список лицензий и сохраняет его в памяти
     * дескриптора, поэтому её лучше не вызывать в цикле, а сохранить значение количества в
     * переменной, и уже её использовать в цикле.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Работает только в режиме `Advanced` конвертера.
     *
     * @return Количество установленных лицензий.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline size_t GetLicenseCount() const;

    /**
     * @brief Запускает асинхронную команду получения количества установленных лицензий.
     *
     * @remark Эта функция читает из контроллера список лицензий и сохраняет его в памяти
     * дескриптора. Работает только в режиме `Advanced` конвертера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_GetLicenseCount() const;

    /**
     * @brief Возвращает количество установленных лицензий.
     *
     * @param[in]  hCommand  Дескриптор команды, которую вернула функция @ref Begin_GetLicenseCount.
     * @param[out] nCount    Количество установленных лицензий.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_GetLicenseCount(ilg_handle hCommand, size_t& nCount);

    /**
     * @brief Возвращает инфо о лицензии из списка, полученного методом @ref GetLicenseCount.
     *
     * @param[in]  nIdx   Позиция в списке.
     * @param[out] rInfo  Информация о лицензии.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetLicenseInfo(size_t nIdx, ilg_license_info& rInfo) const;

    /**
     * @brief Устанавливает лицензию в конвертер.
     *
     * @param[in] pData  Данные лицензии.
     * @param[in] nSize  Количество байт данных лицензии.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Работает только в режиме `Advanced` конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteLicense(const void* pData, size_t nSize);

    /**
     * @brief Устанавливает лицензию в конвертер.
     *
     * @param[in]  nLicenseN   Номер лицензии.
     * @param[in]  pData       Данные лицензии.
     * @param[in]  nSize       Количество байт данных лицензии.
     * @param[out] pInfo       Информация о лицензии. Может быть равен `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Работает только в режиме `Advanced` конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void WriteLicense(uint32_t nLicenseN, const void* pData, size_t nSize,
                             ilg_license_info* pInfo = nullptr);

    /**
     * @brief Запускает асинхронную команду установки лицензии в конвертер.
     *
     * @param[in]  nLicenseN   Номер лицензии.
     * @param[in]  pData       Данные лицензии.
     * @param[in]  nSize       Количество байт данных лицензии.
     *
     * @remark Работает только в режиме `Advanced` конвертера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_WriteLicense(uint32_t nLicenseN, const void* pData, size_t nSize);

    /**
     * @brief Возвращает результат установки лицензии в конвертер.
     *
     * @param[in]  hCommand  Дескриптор команды, которую вернула функция @ref Begin_WriteLicense.
     * @param[out] rInfo     Информация о лицензии.
     *
     * @exception CILGException если произошла ошибка.
     */
    static inline void End_WriteLicense(ilg_handle hCommand, ilg_license_info& rInfo);

    /**
     * @brief Удаляет все лицензии.
     *
     * @param[in] fDisconnect  `true`, после установки лицензии отключается от конвертера , иначе -
     * устанавливает лицензию #5 по умолчанию.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Работает только в режиме `Advanced` конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void ClearLicenses(bool fDisconnect = false);

    /**
     * @brief Запускает асинхронную команду удаления всех лицензий.
     *
     * @param[in]  fDisconnect  `true`, после установки лицензии отключается от конвертера , иначе -
     * устанавливает лицензию #5 по умолчанию.
     *
     * @remark Работает только в режиме `Advanced` конвертера.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_ClearLicenses(bool fDisconnect = false);

    /**
     * @brief Ищет контроллеры.
     *
     * @param[in] fReset  `true`, сбросить старые результаты поиска.
     * @param[in] fForce  `true`, повторить поиск, даже если уже выполнен авто поиск.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void Scan(bool fReset = false, bool fForce = false);

    /**
     * @brief Запускает асинхронную команду поиска конвертеров.
     *
     * @param[in] fReset  `true`, очистить список найденных перед поиском.
     * @param[in] fForce  `true`, повторить поиск, даже если уже выполнен авто поиск.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_Scan(bool fReset = false, bool fForce = false);

    /**
     * @brief Возвращает количество найденных контроллеров.
     *
     * @return Количество найденных контроллеров.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline size_t GetControllerCount() const;

    /**
     * @brief Возвращает инфо о найденном контроллере.
     *
     * @param[in]  nIdx   Позиция в списке найденных контроллеров.
     * @param[out] rInfo  Информация о найденном контроллере.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void GetControllerInfo(size_t nIdx, ilg_controller_info& rInfo) const;

    /**
     * @brief Вкл/выкл режим авто поиска контроллеров.
     *
     * @param[in] fEnable  `true`, включить поиск в реальном времени, иначе - выключить.
     * @param[in] fWait    `true`, ждать завершение операции.
     *
     * @remark Если @p fWait =`true`, то функция не возвращает управление пока ждёт выполнение
     * команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetAutoScan(bool fEnable = true, bool fWait = false);

    /**
     * @brief Запускает асинхронную команду вкл/выкл автоматического сканирования контроллеров.
     *
     * @param[in] fEnable  `true`, включает авто сканирование, иначе - выключает.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetAutoScan(bool fEnable = true);

    /**
     * @brief Возвращает True если авто поиск контроллеров включен.
     *
     * @return `true`, поиск контроллеров включен, иначе - выключен.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline bool GetAutoScan() const;

    /**
     * @brief Устанавливает прошивку в конвертер.
     *
     * @param[in] pData  Данные прошивки.
     * @param[in] nSize  Размер данных прошивки в байтах.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline void SetFirmware(const void* pData, size_t nSize);

    /**
     * @brief Запускает асинхронную команду установки прошивки конвертера.
     *
     * @param[in] pData  Данные прошивки.
     * @param[in] nSize  Размер данных прошивки в байтах.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CAsyncCommand Begin_SetFirmware(const void* pData, size_t nSize);

    /**
     * @brief Возвращает дескриптор подключения к контроллеру.
     *
     * @param[in] nModel  Модель контроллера.
     * @param[in] nSn     Серийный номер контроллера.
     *
     * @return Дескриптор контроллера.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CController GetController(ilg_controller_model nModel, int nSn);

    /**
     * @brief Клонирует дескриптор конвертера.
     * @return Дескриптор конвертера.
     */
    inline CConverter Clone() const;
};

inline CConverter CController::GetConverter() const {
    ilg_handle hNewHandle;
    ILGCheck(ilg_controller_get_converter(m_h, &hNewHandle));
    return CConverter(hNewHandle);
}

inline void CConverter::SetMessageCallback(ilg_converter_message_callback pCallback,
                                           void* pUserData) {
    ILGCheck(ilg_converter_set_message_callback(m_h, pCallback, pUserData));
}

inline void CConverter::EnableMessageQueue(bool fEnable) {
    ILGCheck(ilg_converter_enable_message_queue(m_h, static_cast<ilg_bool>(fEnable)));
}

inline bool CConverter::GetMessage(ilg_converter_msg& nMsg, const void*& pMsgData) const {
    ilg_bool f;
    ILGCheck(ilg_converter_get_message(m_h, &nMsg, &pMsgData, &f));
    return f;
}

inline void CConverter::SetOptions(const ilg_converter_options& rOptions) {
    ILGCheck(ilg_converter_set_options(m_h, &rOptions));
}

inline void CConverter::GetOptions(ilg_converter_options& rOptions) const {
    ILGCheck(ilg_converter_get_options(m_h, &rOptions));
}

inline void CConverter::Connect(bool fReconnect) {
    ILGCheck(ilg_converter_connect(m_h, static_cast<ilg_bool>(fReconnect)));
}

inline CAsyncCommand CConverter::Begin_Connect(bool fReconnect) {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_connect(m_h, static_cast<ilg_bool>(fReconnect), &res.m_h));
    return res;
}

inline void CConverter::Disconnect() {
    ILGCheck(ilg_converter_disconnect(m_h));
}

inline CAsyncCommand CConverter::Begin_Disconnect() {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_disconnect(m_h, &res.m_h));
    return res;
}

inline ilg_connection_status CConverter::GetConnectionStatus() const {
    ilg_connection_status res;
    ILGCheck(ilg_converter_get_connection_status(m_h, &res));
    return res;
}

inline void CConverter::GetConverterInfo(ilg_converter_info& rInfo) const {
    ILGCheck(ilg_converter_get_info(m_h, &rInfo));
}

inline void CConverter::ReadLicenseInfo(ilg_license_info& rInfo, bool fForce) {
    ILGCheck(
        ilg_converter_read_license_info(m_h, &rInfo, ILG_LICENSE_N, static_cast<ilg_bool>(fForce)));
}

inline void CConverter::ReadLicenseInfo(uint32_t nLicenseN, ilg_license_info& rInfo) {
    ILGCheck(ilg_converter_read_license_info(m_h, &rInfo, nLicenseN));
}

inline CAsyncCommand CConverter::Begin_ReadLicenseInfo(uint32_t nLicenseN) {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_read_license_info(m_h, nLicenseN, &res.m_h));
    return res;
}

inline void CConverter::End_ReadLicenseInfo(ilg_handle hCommand, ilg_license_info& rInfo) {
    ILGCheck(ilg_converter_end_read_license_info(hCommand, &rInfo));
}

inline size_t CConverter::GetLicenseCount() const {
    size_t res = 0;
    ILGCheck(ilg_converter_get_license_count(m_h, &res));
    return res;
}

inline CAsyncCommand CConverter::Begin_GetLicenseCount() const {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_get_license_count(m_h, &res.m_h));
    return res;
}

inline void CConverter::End_GetLicenseCount(ilg_handle hCommand, size_t& nCount) {
    ILGCheck(ilg_converter_end_get_license_count(hCommand, &nCount));
}

inline void CConverter::GetLicenseInfo(size_t nIdx, ilg_license_info& rInfo) const {
    ILGCheck(ilg_converter_get_license_info(m_h, nIdx, &rInfo));
}

inline void CConverter::WriteLicense(const void* pData, size_t nSize) {
    ILGCheck(ilg_converter_write_license(m_h, pData, nSize, nullptr));
}

inline void CConverter::WriteLicense(uint32_t nLicenseN, const void* pData, size_t nSize,
                                     ilg_license_info* pInfo) {
    ILGCheck(ilg_converter_write_license(m_h, pData, nSize, pInfo, nLicenseN));
}

inline CAsyncCommand CConverter::Begin_WriteLicense(uint32_t nLicenseN, const void* pData,
                                                    size_t nSize) {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_write_license(m_h, pData, nSize, nLicenseN, &res.m_h));
    return res;
}

inline void CConverter::End_WriteLicense(ilg_handle hCommand, ilg_license_info& rInfo) {
    ILGCheck(ilg_converter_end_write_license(hCommand, &rInfo));
}

inline void CConverter::ClearLicenses(bool fDisconnect) {
    ILGCheck(ilg_converter_clear_licenses(m_h, static_cast<ilg_bool>(fDisconnect)));
}

inline CAsyncCommand CConverter::Begin_ClearLicenses(bool fDisconnect) {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_clear_licenses(m_h, static_cast<ilg_bool>(fDisconnect), &res.m_h));
    return res;
}

inline void CConverter::Scan(bool fReset, bool fForce) {
    ILGCheck(ilg_converter_scan(m_h, static_cast<ilg_bool>(fReset), static_cast<ilg_bool>(fForce)));
}

inline CAsyncCommand CConverter::Begin_Scan(bool fReset, bool fForce) {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_scan(m_h, static_cast<ilg_bool>(fReset),
                                      static_cast<ilg_bool>(fForce), &res.m_h));
    return res;
}

inline size_t CConverter::GetControllerCount() const {
    size_t res = 0;
    ILGCheck(ilg_converter_get_controller_count(m_h, &res));
    return res;
}

inline void CConverter::GetControllerInfo(size_t nIdx, ilg_controller_info& rInfo) const {
    ILGCheck(ilg_converter_get_controller_info(m_h, nIdx, &rInfo));
}

inline void CConverter::SetAutoScan(bool fEnable, bool fWait) {
    ILGCheck(ilg_converter_set_auto_scan(m_h, static_cast<ilg_bool>(fEnable),
                                         static_cast<ilg_bool>(fWait)));
}

inline CAsyncCommand CConverter::Begin_SetAutoScan(bool fEnable) {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_set_auto_scan(m_h, static_cast<ilg_bool>(fEnable), &res.m_h));
    return res;
}

inline bool CConverter::GetAutoScan() const {
    ilg_bool res;
    ILGCheck(ilg_converter_get_auto_scan(m_h, &res));
    return res;
}

inline void CConverter::SetFirmware(const void* pData, size_t nSize) {
    ILGCheck(ilg_converter_set_firmware(m_h, pData, nSize));
}

inline CAsyncCommand CConverter::Begin_SetFirmware(const void* pData, size_t nSize) {
    CAsyncCommand res;
    ILGCheck(ilg_converter_begin_set_firmware(m_h, pData, nSize, &res.m_h));
    return res;
}

inline CController CConverter::GetController(ilg_controller_model nModel, int nSn) {
    CController res;
    ILGCheck(ilg_converter_get_controller(m_h, nModel, nSn, &res.m_h));
    return res;
}

inline CConverter CConverter::Clone() const {
    ilg_handle hNewHandle;
    ILGCheck(ilg_clone_handle(m_h, &hNewHandle));
    return CConverter(hNewHandle);
}

/**
 * \brief Класс для инициализации/финализации библиотеки SDK.
 */
class CILG final {
public:
    /**
     * @brief Конструктор по умолчанию
     *
     * Инициализирует библиотеку с помощью функции @ref ilg_init.
     */
    CILG(bool fInit = true);

    /**
     * @brief Деструктор
     *
     * Завершает работу с библиотекой с помощью @ref ilg_cleanup.
     */
    ~CILG();

    /** Инициализирует библиотеку. */
    void Init();

    /**
     * @brief Возвращает номер версии библиотеки.
     */
    static inline uint32_t GetVersion();

    /**
     * @brief Проверяет версию SDK.
     *
     * @return True, если версия библиотеки совместима с заголовочным файлом.
     */
    static inline bool CheckVersion();

    /**
     * @brief Устанавливает уровень лога отладки.
     *
     * @param[in] nLevel Уровень лога отладки.
     */
    static inline void SetLogLevel(ilg_log_level nLevel);

    /**
     * @brief Устанавливает функцию обратного вызова для лога отладки.
     *
     * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать для передачи
     * сообщений лога отладки.
     * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в try catch.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе вернёт ошибку
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
     */
    static inline void SetLogCallback(ilg_logging_callback pCallback, void* pUserData = nullptr);

    /**
     * @brief Устанавливает функцию обратного вызова для фильтрации портов при поиске считывателей.
     *
     * @param[in] pCallback Указатель на функцию, которую поток поиска считывателей будет вызывать
     * при нахождении порта.
     * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе вернёт ошибку
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
     *
     * @exception CILGException если произошла ошибка `ILG_E_OUTOFMEMORY`, `ILG_E_FAIL`.
     */
    inline void SetFilterPortCallback(ilg_filter_port_callback pCallback,
                                      void* pUserData = nullptr);

    /**
     * @brief Устанавливает глобальные настройки библиотеки.
     *
     * @param[in] rOptions Опции библиотеки.
     *
     * @exception CILGException если произошла ошибка `ILG_E_OUTOFMEMORY`, `ILG_E_FAIL`.
     */
    inline void SetOptions(const ilg_options& rOptions);

    /**
     * @brief Возвращает глобальные настройки библиотеки.
     *
     * @param[out] rOptions Опции библиотеки.
     *
     * @exception CILGException если произошла ошибка `ILG_E_OUTOFMEMORY`, `ILG_E_FAIL`.
     */
    inline void GetOptions(ilg_options& rOptions) const;

    /**
     * @brief Создаёт дескриптор поиска конвертеров.
     *
     * @remark Эта функция создаёт поток поиска конвертеров, если ещё не создан (один поток на
     * библиотеку). Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * конвертеров.
     *
     * @return Класс поиска конвертеров.
     *
     * @exception CILGException если произошла ошибка `ILG_E_OUTOFMEMORY`, `ILG_E_FAIL`,
     * `ILG_E_BLOCKING_CALL_NOT_ALLOWED`, `ILG_E_OUT_OF_RESOURCES`.
     */
    inline CConverterSearch GetSearch();

    /**
     * @brief Создаёт дескриптор конвертера.
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     *
     * @remark Эта функция создаёт поток конвертера, если ещё не создан (один поток на порт).
     * Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
     * Функция позволяет создать более одного дескриптора для одного порта.
     *
     * @return Класс конвертеров.
     *
     * @exception CILGException если произошла ошибка.
     */
    inline CConverter GetConverter(ilg_port_type nPortType, const char* pszPortName);

protected:
    /** true, библиотека инициализирована. */
    bool m_fInit = false;
};

inline uint32_t CILG::GetVersion() {
    return ilg_get_version();
}

inline bool CILG::CheckVersion() {
    return ilg_check_version();
}

inline void CILG::SetLogLevel(ilg_log_level nLevel) {
    ILGCheck(ilg_set_log_level(nLevel));
}

inline void CILG::SetLogCallback(ilg_logging_callback pCallback, void* pUserData) {
    ILGCheck(ilg_set_log_callback(pCallback, pUserData));
}

inline void CILG::SetFilterPortCallback(ilg_filter_port_callback pCallback, void* pUserData) {
    ILGCheck(ilg_set_filter_port_callback(pCallback, pUserData));
}

inline void CILG::SetOptions(const ilg_options& rOptions) {
    ILGCheck(ilg_set_options(&rOptions));
}

inline void CILG::GetOptions(ilg_options& rOptions) const {
    ILGCheck(ilg_get_options(&rOptions));
}

inline CConverterSearch CILG::GetSearch() {
    CConverterSearch res;
    ILGCheck(ilg_get_search(&res.m_h));
    return res;
}

inline CConverter CILG::GetConverter(ilg_port_type nPortType, const char* pszPortName) {
    CConverter res;
    ILGCheck(ilg_get_converter(nPortType, pszPortName, &res.m_h));
    return res;
}

/**
 * @brief Возвращает true если бит bitN установлен в числе val.
 *
 * @param[in] val  Число, в котором проверяется бит.
 * @param[in] bitN Номер бита (отсчёт от 0).
 */
#define GET_BIT(val, bitN) (bool)((val >> (bitN)) & 1)
/**
 * @brief Устанавливает/снимает бит bitN в числе val.
 *
 * @param[in] val  Число, в котором изменяется бит.
 * @param[in] bitN Номер бита (отсчёт от 0).
 * @param[in] On   true, установить бит, иначе - снять.
 */
#define SET_BIT(val, bitN, On) \
    { (On) ? ((val) |= (1 << (bitN))) : ((val) &= ~(1 << (bitN))); }

/** Возвращает текущее время. */
inline std::chrono::steady_clock::time_point now() {
    return std::chrono::steady_clock::now();
}

/**
 * @brief Возвращает интервал времени в миллисекундах от времени start до текущего времени.
 *
 * @param[in] start Начальное время.
 */
template <class result_t = std::chrono::milliseconds, class clock_t = std::chrono::steady_clock,
          class duration_t = std::chrono::milliseconds>
inline auto since(std::chrono::time_point<clock_t, duration_t> const& start) {
    return std::chrono::duration_cast<result_t>(clock_t::now() - start);
}

/// Названия типов портов конвертеров
extern const char* kPortTypeNames[ILG_PORT_TYPE_SIZE];
/// Названия моделей конвертеров
extern const char* kConverterModelNames[ILG_CONVERTER_MODEL_SIZE];
/// Названия моделей контроллеров
extern const char* kControllerModelNames[ILG_CONTROLLER_MODEL_SIZE];
/// Названия режимов конвертеров
extern const char* kConverterModeNames[ILG_CONVERTER_MODE_SIZE];

/**
 * @brief Преобразует версию Sdk Readers в строку.
 *
 * @param[in] nVersion  Версия SDK.
 */
std::string SdkVersionToStr(uint32_t nVersion);

/**
 * @brief Преобразует версию конвертера/контроллера в строку.
 *
 * @param[in] nVersion  Версия конвертера/контроллера.
 */
std::string VersionToStr(uint32_t nVersion);

/**
 * @brief Преобразует время в строку.
 *
 * @param[in] tTime  Время.
 */
std::string TimeToStr(const int64_t& tTime);

/**
 * @brief Преобразует номер ключа в строку.
 *
 * @param[in] rNumber    Номер ключа.
 * @param[in] nKeyFlags  Флаги ключа.
 * @param[in] fWiegand   `true`, преобразуется в формат Em-Marine, иначе Dallas.
 *
 * @return Строку с номером ключа.
 */
std::string KeyNumberToStr(const ilg_key_number& rNumber, uint32_t nKeyFlags, bool fWiegand);
/**
 * Пытается преобразовать строку в номер ключа.
 *
 * @param[in]  pStr       Строка.
 * @param[out] pNumber    Номер ключа.
 * @param[out] pKeyFlags  Флаги ключа.
 *
 * @return `true`, преобразование выполнено успешно.
 */
bool TryStrToKeyNum(const char* pStr, ilg_key_number* pNumber, uint8_t* pKeyFlags);

/**
 * @brief Преобразует номер ключа в строку с кодом клавиатуры.
 *
 * @param[in] rNumber   Номер ключа.
 * @param[in] nDualIdx  Номер двойного номера: 0 первый двойной номер, 1 второй, -1 номер не
 * двойной, использовать весь номер.
 * @param[in] chError   Символ, на который заменяется некорректная цифра кода клавиатуры.
 *
 * @return Строка с кодом клавиатуры.
 */
std::string KeybCodeToStr(const ilg_key_number& rNumber, int nDualIdx = -1, char chError = '_');

/**
 *  @brief Пытается преобразовать строку с кодом клавиатуры в номер ключа.
 *
 * @param[in]  pStr     Строка.
 * @param[out] pNumber  Номер ключа.
 *
 * @return `true`, преобразование выполнено успешно.
 */
bool TryParseKeybCodeStr(const char* pStr, ilg_key_number* pNumber);

};  // namespace ilg
