#include <exception>
#include <fstream>   // для std::fstream
#include <iomanip>   // для std::put_time
#include <iostream>  // для std::cout и std::cin

#include "ilg_cpp_helpers.h"
#include "ilguard.h"

// #define ILG_LOG  // Раскомментируйте, чтобы включить показ отладочных сообщений
#define ILG_LOG_FILE  // Писать лог в файл

using namespace ilg;

#ifdef ILG_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilguard.log";  // Путь к лог файлу

void ILG_CALL LogCallback(ilg_log_level level, const char* pContext, const char* pMessage, void*) {
#if 1  // Запись в файл
    std::fstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void ILG_CALL MessageCallback(ilg_search_msg nMsg, const void* pMsgData, void*) {
    try {
        switch (nMsg) {
        case ILG_SEARCH_MSG_CONVERTER_FOUND:  // Конвертер найден
        case ILG_SEARCH_MSG_CONVERTER_LOST:   // Конвертер потерян
            {
                auto pInfo = static_cast<const ilg_converter_info*>(pMsgData);
                std::stringstream ss;
                if (pInfo->nModel != ILG_CONVERTER_MODEL_UNKNOWN)
                    ss << kConverterModelNames[pInfo->nModel];
                if (pInfo->nSn != -1)
                    ss << " с/н:" << pInfo->nSn;
                if (pInfo->nFwVersion != 0)
                    ss << " прошивка:" << VersionToStr(pInfo->nFwVersion);
                if (pInfo->nFwBuildDate != 0)
                    ss << " сборка " << TimeToStr(pInfo->nFwBuildDate);

                if (ILG_SEARCH_MSG_CONVERTER_FOUND == nMsg)
                    std::cout << "{!} Конвертер найден (" << pInfo->pszPortName << "): " << ss.str()
                              << std::endl;
                else
                    std::cout << "{!} Конвертер потерян (" << pInfo->pszPortName
                              << "): " << ss.str() << std::endl;
                break;
            }

        case ILG_SEARCH_MSG_LIST_CHANGED:  // Список конвертеров изменился
            std::cout << "{!} Список конвертеров изменился" << std::endl;
            break;

        default:
            break;
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
}

int main() {
    try {
#ifdef ILG_LOG
#ifdef ILG_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        CILG::SetLogCallback(LogCallback);
        CILG::SetLogLevel(ILG_LOG_LEVEL_DEBUG);
#endif
        CILG oILG;
        CConverterSearch oSearch(oILG.GetSearch());
#if 0
        // Выбираем типы конвертеров для поиска
        ilg_search_options rOptions;
        oSearch.GetOptions(rOptions);
        rOptions.nConverterTypes =
            ILG_CONVERTER_ILUSB | ILG_CONVERTER_SERVER | ILG_CONVERTER_CLIENT;
        oSearch.SetOptions(rOptions);
        // Устанавливаем TCP-порты конвертеров в режиме "Клиент"
        const uint16_t aListenPorts[] = {25000};
        oSearch.SetListenPorts(aListenPorts, std::size(aListenPorts));
#endif
        // Подписываемся на уведомления о нахождении/потери конвертеров
        oSearch.SetMessageCallback(MessageCallback);
        // Включаем авто поиск конвертеров (по умолчанию выключен)
        oSearch.SetAutoScan();
        std::cout << "Ожидание подключения/отключения конвертеров..." << std::endl;

        std::cout << "Нажмите <Enter> для выхода..." << std::endl;
        std::cin.get();
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }

    return 0;
}
