#include <exception>
#include <fstream>   // для std::fstream
#include <iomanip>   // для std::put_time
#include <iostream>  // для std::cout и std::cin
#include <string>
#include <vector>

#include "ilg_cpp_helpers.h"
#include "ilguard/ilguard.h"

// #define ILG_LOG  // Раскомментируйте, чтобы включить показ отладочных сообщений
#define ILG_LOG_FILE  // Писать лог в файл

using namespace ilg;

// Глобальные переменные:
uint8_t g_nBankCount = 0;  // Количество банков ключей контроллера
uint32_t g_nCtrFlags = 0;  // Флаги контроллера
size_t g_nMaxKeys = 0;     // Ёмкость банка ключей

// Названия типов ключей
const char* kKeyTypeNames[] = {"Простой", "Блокирующий", "Мастер"};
#ifdef ILG_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilguard.log";  // Путь к лог файлу

void ILG_CALL LogCallback(ilg_log_level level, const char* pContext, const char* pMessage, void*) {
#if 1  // Запись в файл
    std::fstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

// Показывает текущий список ключей контроллера
void ShowKeys(CController& oController) {
    bool fWiegand = (g_nCtrFlags & ILG_CTR_F_WIEGAND) != 0;
    std::cout << "Ключи:" << std::endl;
    auto tStartTime = now();
    std::vector<ilg_key> oKeys;
    ilg_key* pKey;
    for (uint8_t i = 0; i < g_nBankCount; i++) {
        std::cout << "Банк " << (uint)i << ':' << std::endl;
        auto nTopIdx = oController.ReadKeyTopIdx(i);
        if (nTopIdx < 0) {
            std::cout << "Некорректный указатель верхней границы ключей банка " << i << std::endl;
            nTopIdx = static_cast<ssize_t>(g_nMaxKeys);
        }
        if (0 == nTopIdx)
            std::cout << "Список пустой" << std::endl;
        else {
            oKeys.resize(static_cast<size_t>(nTopIdx));
            std::cout << "Чтение..." << std::endl;
            oController.ReadKeys(i, 0, oKeys.data(), oKeys.size());
            for (size_t j = 0; j < oKeys.size(); j++) {
                pKey = &oKeys[j];
                if (pKey->fErased)
                    std::cout << j << " пустой" << std::endl;
                else
                    std::cout << j << " " << KeyNumberToStr(pKey->rNumber, pKey->nFlags, fWiegand)
                              << ", " << kKeyTypeNames[pKey->nType] << ", доступ: " << std::hex
                              << std::showbase << (uint)pKey->nAccess << std::dec << std::endl;
            }
        }
        std::cout << "------------" << std::endl;
    }
    std::cout << "Выполнено за " << since(tStartTime).count() << " мс" << std::endl;
}

// Добавляет или изменяет один ключ в контроллере
void SetKey(CController& oController) {
    std::cout << "Введите номер банка, позицию ключа (-1 верхняя граница), тип (1 Простой, 2 "
                 "Блокирующий, 3 "
                 "Мастер), доступ (16-ричное), номер (-1 последний ключ):"
              << std::endl;

    int nIdx;
    uint nBankN, nType, nAccess;
    std::string sKeyNumber;
    std::cin >> nBankN >> nIdx >> nType >> std::hex >> nAccess >> std::quoted(sKeyNumber);
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
    if (std::cin.fail()) {
        std::cout << "Неправильный ввод" << std::endl;
        return;
    }

    if (-1 == nIdx) {
        auto nTopIdx = oController.ReadKeyTopIdx(static_cast<uint8_t>(nBankN));
        if (nTopIdx < 0) {
            std::cout << "Некорректный указатель верхней границы ключей банка " << nBankN
                      << std::endl;
            nTopIdx = static_cast<ssize_t>(g_nMaxKeys);
        }

        if (nTopIdx == static_cast<ssize_t>(g_nMaxKeys)) {
            std::cout << "Нет свободного места в конце списка" << std::endl;
            return;
        }
        nIdx = static_cast<int>(nTopIdx);
    }
    ilg_key rKey = {};
    if (sKeyNumber == "-1") {
        ilg_rtc_params rRtc;
        oController.ReadRtcParams(rRtc);
        rKey.rNumber = rRtc.rLastKey;
    }
    else if (!TryStrToKeyNum(sKeyNumber.c_str(), &rKey.rNumber, &rKey.nFlags)) {
        std::cout << "Неправильный ввод" << std::endl;
        return;
    }
    rKey.fErased = false;
    rKey.nType = static_cast<ilg_key_type>(nType - 1);
    rKey.nAccess = static_cast<uint8_t>(nAccess);
    std::cout << "Запись..." << std::endl;
    auto tStartTime = now();
    oController.WriteKeys(static_cast<uint8_t>(nBankN), static_cast<size_t>(nIdx), &rKey, 1);
    std::cout << "Выполнено за " << since(tStartTime).count() << " мс" << std::endl;
}

// Очищает список ключей в контроллере
void ClearKeys(CController& oController) {
    std::cout << "Очистка..." << std::endl;
    auto tStartTime = now();
    for (uint8_t i = 0; i < g_nBankCount; ++i) {
        auto nTopIdx = oController.ReadKeyTopIdx(i);
        if (0 == nTopIdx)
            continue;
        size_t nCount;
        if (nTopIdx < 0) {
            std::cout << "Некорректный указатель верхней границы ключей банка " << i << std::endl;
            nCount = g_nMaxKeys;
        }
        else
            nCount = static_cast<size_t>(nTopIdx);
        oController.EraseKeys(i, 0, nCount);
    }
    std::cout << "Выполнено за " << since(tStartTime).count() << " мс" << std::endl;
}

// Подключается к конвертеру
bool DoConnectToConverter(CILG& oILR, CConverter& oConverter) {
    // Ищем конвертеры
    CConverterSearch oSearch(oILR.GetSearch());
    std::cout << "Поиск конвертеров..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetConverterCount();
    if (0 == nCount) {
        std::cout << "Конвертер не найден" << std::endl;
        return false;
    }
    std::cout << "Найдено конвертеров: " << nCount << std::endl;

    std::cout << std::endl << "Выберите конвертер:" << std::endl;
    for (size_t i = 0; i < nCount; i++) {
        ilg_converter_info rInfo;
        oSearch.GetConverterInfo(i, rInfo);

        std::stringstream ss;
        if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
            ss << kConverterModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
            ss << " режим: " << kConverterModeNames[rInfo.nMode];
        std::cout << 1 + i << ". " << rInfo.pszPortName << " [" << rInfo.pszConnect
                  << "]: " << ss.str() << std::endl;
    }
    std::cout << "0 - Выйти из программы" << std::endl;

    int nCommand;
    std::cin >> nCommand;
    if (std::cin.fail()) {
        std::cin.clear();
        nCommand = -1;
    }
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');

    if ((nCommand <= 0) || (static_cast<size_t>(nCommand) > nCount))
        return false;

    ilg_converter_info rInfo;
    oSearch.GetConverterInfo(static_cast<size_t>(nCommand - 1), rInfo);

    std::cout << "Подключение к конвертеру [" << kPortTypeNames[rInfo.nPortType] << ": "
              << rInfo.pszPortName << "]..." << std::endl;
    oConverter = oILR.GetConverter(rInfo.nPortType, rInfo.pszPortName);
    ilg_converter_options rOptions;
    oConverter.GetOptions(rOptions);
    rOptions.nConnectModel = rInfo.nModel;
    oConverter.SetOptions(rOptions);
    // Подключаемся к конвертеру
    oConverter.Connect();
    // Получаем информацию о конвертере
    oConverter.GetConverterInfo(rInfo);
    std::stringstream ss;
    if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
        ss << kConverterModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н:" << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
    if (rInfo.nFwBuildDate != 0)
        ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
    if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
        ss << " режим: " << kConverterModeNames[rInfo.nMode];
    std::cout << "Конвертер успешно подключён [" << ss.str() << ']' << std::endl;
    return true;
}

// Подключается к контроллеру
bool DoConnectToController(CConverter& oConverter, CController& oController) {
    // Поиск контроллеров
    int nCommand;
    while (true) {
        std::cout << "Выберите контроллер:" << std::endl;
        oConverter.Scan();
        auto nCount = oConverter.GetControllerCount();
        for (size_t i = 0; i < nCount; i++) {
            ilg_controller_info rInfo;
            oConverter.GetControllerInfo(i, rInfo);
            std::cout << i + 1 << ". #" << (uint)rInfo.nAddress << ": "
                      << kControllerModelNames[rInfo.nModel] << " с/н:" << rInfo.nSn
                      << " прошивка:" << VersionToStr(rInfo.nFwVersion) << std::endl;
        }
        if (0 == nCount)
            std::cout << "1 - Искать снова" << std::endl;
        std::cout << "0 - Выйти из программы" << std::endl;

        std::cin >> nCommand;
        if (std::cin.fail()) {
            std::cin.clear();
            nCommand = -1;
        }
        std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');

        if (0 == nCommand)
            return false;
        if ((nCommand >= 1) && (static_cast<size_t>(nCommand) <= nCount))
            break;
    }
    ilg_controller_info rInfo;
    oConverter.GetControllerInfo(static_cast<size_t>(nCommand - 1), rInfo);
    oController = oConverter.GetController(rInfo.nModel, rInfo.nSn);
    // Подключаемся к контроллеру
    std::cout << "Подключение к контроллеру [" << kControllerModelNames[rInfo.nModel] << ": "
              << rInfo.nSn << "]..." << std::endl;
    oController.Connect();
    // Получаем информацию о контроллере
    oController.GetControllerInfo(rInfo);
    std::cout << "Контроллер успешно подключён [#" << (uint)rInfo.nAddress << ' '
              << kControllerModelNames[rInfo.nModel] << " с/н:" << rInfo.nSn
              << " прошивка:" << VersionToStr(rInfo.nFwVersion) << ']' << std::endl;
    // Выключаем авто поиск контроллеров (не обязательно)
    oConverter.SetAutoScan(false);
    return true;
}

int main() {
    try {
#ifdef ILG_LOG
#ifdef ILG_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        // Включаем лог отладки
        CILG::SetLogCallback(LogCallback);
        CILG::SetLogLevel(ILG_LOG_LEVEL_DEBUG);
#endif

        CILG oILG;

        // Подключаемся к конвертеру
        CConverter oConverter;
        if (!DoConnectToConverter(oILG, oConverter))
            return 0;

        // Подключаемся к контроллеру
        CController oController;
        if (!DoConnectToController(oConverter, oController))
            return 0;

        // Получаем информацию о контроллере
        ilg_controller_info rInfo;
        oController.GetControllerInfo(rInfo);
        g_nBankCount = (rInfo.nCtrFlags & ILG_CTR_F_TWOBANKS) ? 2 : 1;
        g_nCtrFlags = rInfo.nCtrFlags;
        g_nMaxKeys = (rInfo.nBankSize - 0xC0) / 8;
        if (rInfo.nCtrFlags & ILG_CTR_F_X2)
            g_nMaxKeys *= 2;

        while (true) {
            std::cout << "-----" << std::endl;
            std::cout << "Введите номер команды:" << std::endl;
            std::cout << "1 - Показать ключи" << std::endl;
            std::cout << "2 - Установить ключ..." << std::endl;
            std::cout << "3 - Очистить список ключей" << std::endl;
            std::cout << "0 - Выйти из программы" << std::endl;

            int nCommand;
            std::cin >> nCommand;
            if (std::cin.fail()) {
                std::cin.clear();
                nCommand = -1;
            }
            std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');

            switch (nCommand) {
            case 0:
                return 0;

            case 1:
                ShowKeys(oController);
                break;

            case 2:
                SetKey(oController);
                break;

            case 3:
                ClearKeys(oController);
                break;

            default:
                std::cout << "Неправильный ввод" << std::endl;
                break;
            }
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
