#include "CMonitorDialog.h"

#include <iomanip>  // для std::put_time
#include <sstream>  // для std::stringstream

#include "utils.h"

CMonitorDialog::CMonitorDialog() :
    m_fWiegand(false),
    m_nMaxEvents(0),
    m_nReadIdx(-1),
    m_nWriteIdx(-1),
    m_Box(Gtk::Orientation::VERTICAL),
    m_ClearButton("Очистить"),
    m_OpenInButton("Открыть для входа"),
    m_OpenOutButton("Открыть для выхода"),
    m_LastKeyLabel("Последний ключ:") {
    set_destroy_with_parent(true);
    set_title("Монитор событий");
    set_default_size(800, 480);

    m_Box.set_margin(5);
    set_child(m_Box);
    m_Box.append(m_ScrolledWindow);
    m_Box.append(m_BottomBox);

    // Add the ColumnView, inside a ScrolledWindow, with the button underneath:
    m_ScrolledWindow.set_child(m_ColumnView);
    // Only show the scrollbars when they are necessary:
    m_ScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_ScrolledWindow.set_expand();

    // Create the List model:
    m_ListStore = Gio::ListStore<CModelColumns>::create();

    // Set list model and selection model.
    auto selection_model = Gtk::SingleSelection::create(m_ListStore);
    selection_model->set_autoselect(false);
    selection_model->set_can_unselect(true);
    m_ColumnView.set_model(selection_model);
    m_ColumnView.add_css_class("data-table");  // high density table

    // Add the ColumnView's columns:

    // Столбец "№"
    auto factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMonitorDialog::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMonitorDialog::on_bind_idx));
    auto column = Gtk::ColumnViewColumn::create("№", factory);
    m_ColumnView.append_column(column);

    // Столбец "Дата и время"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMonitorDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMonitorDialog::on_bind_time));
    column = Gtk::ColumnViewColumn::create("Дата и время", factory);
    m_ColumnView.append_column(column);

    // Столбец "Событие"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMonitorDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMonitorDialog::on_bind_event));
    column = Gtk::ColumnViewColumn::create("Событие", factory);
    m_ColumnView.append_column(column);

    // Столбец "Направление"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMonitorDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMonitorDialog::on_bind_direction));
    column = Gtk::ColumnViewColumn::create("Направление", factory);
    m_ColumnView.append_column(column);

    // Столбец "Ключ"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMonitorDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMonitorDialog::on_bind_key));
    column = Gtk::ColumnViewColumn::create("Ключ", factory);
    m_ColumnView.append_column(column);

    // Столбец "Детали"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMonitorDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMonitorDialog::on_bind_details));
    column = Gtk::ColumnViewColumn::create("Детали", factory);
    m_ColumnView.append_column(column);

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.append(m_ClearButton);
    m_BottomBox.append(m_OpenInButton);
    m_BottomBox.append(m_OpenOutButton);
    m_BottomBox.append(m_LastKeyLabel);
    m_BottomBox.append(m_LastKeyValueLabel);

    m_LastKeyLabel.set_hexpand();
    m_LastKeyLabel.set_halign(Gtk::Align::END);
    m_LastKeyValueLabel.set_selectable();

    m_ClearButton.signal_clicked().connect(sigc::mem_fun(*this, &CMonitorDialog::on_button_clear));
    m_OpenInButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CMonitorDialog::on_button_open_in));
    m_OpenOutButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CMonitorDialog::on_button_open_out));

    m_oDisp.connect(sigc::mem_fun(*this, &CMonitorDialog::on_ilg));

    signal_destroy().connect(sigc::mem_fun(*this, &CMonitorDialog::on_destroy));
}

CMonitorDialog::~CMonitorDialog() {
    m_oController = nullptr;
}

void CMonitorDialog::Init(const ilg::CController& oController) {
    m_oController = oController.Clone();

    ilg_controller_info rInfo;
    m_oController.GetControllerInfo(rInfo);
    m_fWiegand = (rInfo.nCtrFlags & ILG_CTR_F_WIEGAND) != 0;
    m_nMaxEvents = (rInfo.nBankSize / 8);
    m_oController.EnableMessageQueue();
    m_oController.SetMessageCallback(&on_ilg_message, this);

    ilg_rtc_params rRtc;
    m_oController.ReadRtcParams(rRtc);
    // m_nReadIdx = rRtc.nEventReadIdx;
    m_nWriteIdx = rRtc.nEventWriteIdx;
    m_nReadIdx = m_nWriteIdx;
    m_LastKeyValueLabel.set_label(ilg::KeyNumberToStr(rRtc.rLastKey, 0, m_fWiegand));
    if (m_nReadIdx != m_nWriteIdx)
        StartReadEvents();
}

void CMonitorDialog::on_setup_label(const Glib::RefPtr<Gtk::ListItem>& list_item,
                                    Gtk::Align halign) {
    list_item->set_child(*Gtk::make_managed<Gtk::Label>("", halign));
}

void CMonitorDialog::on_bind_idx(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(Glib::ustring::sprintf("%u", col->m_nIdx));
}

void CMonitorDialog::on_bind_time(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    std::stringstream ss;
    if (col->m_tTime != 0) {
        auto tmb = std::localtime(&col->m_tTime);
        ss << std::put_time(tmb, "%d-%m-%Y %H:%M:%S");
    }
    label->set_text(ss.str());
}

void CMonitorDialog::on_bind_event(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(kEventTypeNames[col->m_nType]);
}

void CMonitorDialog::on_bind_direction(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(kDirectionNames[col->m_nDirection]);
}

void CMonitorDialog::on_bind_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;

    auto key = col->m_property_number.get_value();
    if (key.m_fEmpty)
        label->set_text("");
    else
        label->set_text(ilg::KeyNumberToStr(key.m_Number, key.m_nFlags, m_fWiegand));
    Glib::Binding::bind_property_value(col->m_property_number.get_proxy(), label->property_label(),
                                       Glib::Binding::Flags::DEFAULT,
                                       sigc::mem_fun(*this, &CMonitorDialog::number_transform_to));
}

bool CMonitorDialog::number_transform_to(const GValue* a, GValue* b) {
    auto pSrc = (const MyKeyNumber*)a->data->v_pointer;
    if (b->g_type != G_TYPE_STRING)
        return false;
    if (pSrc->m_fEmpty)
        g_value_set_string(b, "");
    else
        g_value_set_string(b,
                           ilg::KeyNumberToStr(pSrc->m_Number, pSrc->m_nFlags, m_fWiegand).c_str());
    return true;
}

void CMonitorDialog::on_bind_details(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    std::stringstream ss;
    switch (col->m_nFormat) {
    case ILG_EVENT_FORMAT_PASSAGE:
        if (col->m_nKeyIdx != -1)
            ss << "Позиция ключа " << std::dec << col->m_nKeyIdx << " (банк " << col->m_nKeyBankN
               << ")";
        break;
    case ILG_EVENT_FORMAT_CONTROLLER_MODE:
        ss << kControllerModeNames[col->m_nMode] << "; Флаги: " << std::hex << std::showbase
           << (uint)col->m_nModeFlags << ", триггер: " << std::dec << (uint)col->m_nModeTrigger;
        break;
    case ILG_EVENT_FORMAT_STATE:
        {
            bool fAddSemicolon = true;
            switch (col->m_nType) {
            case ILG_EVENT_ELECTROCONTROL:
                if (col->m_nModeFlags & 1)
                    ss << "Питание включено";
                else
                    ss << "Питание выключено";
                break;

            case ILG_EVENT_FIRE:
                if (col->m_nModeFlags & 1)
                    ss << "Пожарный режим включен";
                else
                    ss << "Пожарный режим выключен";
                break;

            case ILG_EVENT_SECURITY:
                if (col->m_nModeFlags & 1)
                    ss << "Охранный режим включен";
                else
                    ss << "Охранный режим выключен";
                ss << "; ";
                if (col->m_nModeFlags & 2)
                    ss << "Тревога включена";
                else
                    ss << "Тревога выключена";
                break;

            default:
                fAddSemicolon = false;
                break;
            }
            if (fAddSemicolon)
                ss << "; ";
            ss << "Флаги: " << std::hex << std::showbase << (uint)col->m_nModeFlags
               << ", триггер: " << std::dec << (uint)col->m_nModeTrigger;
            break;
        }
    }
    label->set_text(ss.str());
}

void CMonitorDialog::on_button_clear() {
    m_ListStore->remove_all();
}

void CMonitorDialog::on_button_open_in() {
    m_oController.OpenDoor(false);
}

void CMonitorDialog::on_button_open_out() {
    m_oController.OpenDoor(true);
}

void CMonitorDialog::on_destroy() {
    m_oController = nullptr;
}

void ILG_CALL CMonitorDialog::on_ilg_message(ilg_controller_msg nMsg, const void* pMsgData,
                                             void* pUserData) {
    auto p = (CMonitorDialog*)pUserData;
    p->m_oDisp.emit();
}

void CMonitorDialog::on_ilg() {
    ilg_controller_msg nMsg;
    const void* pMsgData;
    while (m_oController.GetMessage(nMsg, pMsgData)) {
        switch (nMsg) {
        case ILG_CONTROLLER_MSG_COMMAND_FINISH:
            on_ilg_command_finish((ilg_handle)pMsgData);
            break;

        case ILG_CONTROLLER_MSG_RTC_CHANGED:
            on_ilg_rtc_changed();
            break;
        }
    }
}

void CMonitorDialog::on_ilg_command_finish(ilg_handle hCommand) {
    if (hCommand != m_oCommand.Get())
        return;
    ilg::CAsyncCommand oCommand(std::move(m_oCommand));

    const uint64_t* pList;
    size_t nCount;
    m_oController.End_ReadEvents(oCommand, pList, nCount);
    AddEvents(pList, nCount);

    if (m_nReadIdx != m_nWriteIdx)
        StartReadEvents();
}

void CMonitorDialog::on_ilg_rtc_changed() {
    ilg_rtc_params rRtc;
    m_oController.GetRtcParams(rRtc);
    m_nWriteIdx = rRtc.nEventWriteIdx;
    m_LastKeyValueLabel.set_label(ilg::KeyNumberToStr(rRtc.rLastKey, 0, m_fWiegand));
    if (m_oCommand == nullptr)
        StartReadEvents();
}

void CMonitorDialog::StartReadEvents() {
    if (-1 == m_nWriteIdx)
        return;
    if (-1 == m_nReadIdx)
        m_nReadIdx = m_nWriteIdx;

    size_t nNewCount;
    if (m_nWriteIdx >= m_nReadIdx)
        nNewCount = static_cast<size_t>(m_nWriteIdx - m_nReadIdx);
    else
        nNewCount = static_cast<size_t>(m_nMaxEvents - m_nReadIdx + m_nWriteIdx);
    if (0 == nNewCount)
        return;
    m_oCommand = m_oController.Begin_ReadEvents(static_cast<size_t>(m_nReadIdx), nNewCount);
}

void CMonitorDialog::AddEvents(const uint64_t* pList, size_t nCount) {
    time_t tEventTime;          // Дата и время события
    ilg_event_type nEventType;  // Тип события
    ilg_event_format nFormat;   // Формат записи события
    ilg_direction nDirection;   // Направление
    bool fKeyEmpty;             // true, ключ пустой (нет ключа)
    ilg_key_number rKeyNumber;  // Номер ключа
    uint8_t nKeyFlags;          // Флаги ключа
    uint8_t nModeFlags;         // Флаги режима
    uint8_t nModeTrigger;       // Триггер режима
    ilg_controller_mode nMode;  // Режим контроллера
    uint8_t nKeyBankN;          // Номер банка ключей
    ssize_t nKeyIdx;            // Позиция в банке ключей
    ilg_controller_time rTime;
    ilg_key rKey;
    time_t tNow = time(nullptr);
    auto tmb = std::localtime(&tNow);

    auto pEnd = (pList + nCount);
    for (auto p = pList; p != pEnd; p++) {
        auto& nRawEvent = *p;
        nEventType = m_oController.DecodeEventType(nRawEvent, &nFormat);
        tEventTime = 0;
        nDirection = ILG_DIRECTION_UNKNOWN;
        memset(&rTime, 0, sizeof(rTime));
        fKeyEmpty = true;

        switch (nFormat) {
        case ILG_EVENT_FORMAT_PASSAGE:
            m_oController.DecodePassageEvent(nRawEvent, rTime, nDirection, nKeyBankN, nKeyIdx);
            if (nKeyIdx != -1) {
                m_oController.ReadKeys(nKeyBankN, nKeyIdx, &rKey, 1);
                fKeyEmpty = false;
            }
            break;

        case ILG_EVENT_FORMAT_TIME:
            m_oController.DecodeTimeEvent(nRawEvent, rTime);
            break;

        case ILG_EVENT_FORMAT_CONTROLLER_MODE:
            m_oController.DecodeControllerModeEvent(nRawEvent, rTime, nMode, nModeFlags,
                                                    nModeTrigger);
            break;

        case ILG_EVENT_FORMAT_KEY_NUMBER:
            m_oController.DecodeKeyNumber(nRawEvent, rKeyNumber);
            fKeyEmpty = false;
            break;
        }
        if (rTime.nMonth != 0) {
            struct tm atm;
            atm.tm_sec = rTime.nSecond;
            atm.tm_min = rTime.nMinute;
            atm.tm_hour = rTime.nHour;
            atm.tm_mday = rTime.nDay;
            atm.tm_mon = rTime.nMonth - 1;  // tm_mon is 0 based
            atm.tm_year = tmb->tm_year;     // tm_year is 1900 based
            atm.tm_isdst = -1;
            tEventTime = mktime(&atm);
            if (-1 == tEventTime)
                tEventTime = 0;
            if (tEventTime != 0) {
                // Если (дата события + 30 дней) > текущей даты,
                if ((tEventTime + (60 * 60 * 24 * 30)) > tNow) {
                    // Вычитаем 1 год
                    tEventTime -= 60 * 60 * 24 * (365 + (IsLeapYear(1900 + atm.tm_year) ? 1 : 0));
                }
            }
        }

        // m_ListStore->append(CModelColumns::create(
        //     m_nReadIdx, tEventTime, nEventType, nFormat, nDirection, fKeyEmpty, rKeyNumber,
        //     nKeyFlags, nModeFlags, nModeTrigger, nMode, nKeyBankN, nKeyIdx));
        m_ListStore->insert(
            0, CModelColumns::create(m_nReadIdx, tEventTime, nEventType, nFormat, nDirection,
                                     fKeyEmpty, rKeyNumber, nKeyFlags, nModeFlags, nModeTrigger,
                                     nMode, nKeyBankN, nKeyIdx));

        if (++m_nReadIdx == m_nMaxEvents)
            m_nReadIdx = 0;
    }
}
