#include <exception>
#include <fstream>   // для std::fstream
#include <iomanip>   // для std::put_time
#include <iostream>  // для std::cout и std::cin

#include "ilg_cpp_helpers.h"
#include "ilguard/ilguard.h"

// #define ILG_LOG  // Раскомментируйте, чтобы включить показ отладочных сообщений
#define ILG_LOG_FILE  // Писать лог в файл

using namespace ilg;

#ifdef ILG_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilguard.log";  // Путь к лог файлу

void ILG_CALL LogCallback(ilg_log_level level, const char* pContext, const char* pMessage, void*) {
#if 1  // Запись в файл
    std::fstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void ILG_CALL MessageCallback(ilg_converter_msg nMsg, const void*, void* pUserData) {
    if (ILG_CONVERTER_MSG_CONNECTION_CHANGED == nMsg) {
        try {
            auto pConverter = static_cast<ilg::CConverter*>(pUserData);
            auto nStatus = pConverter->GetConnectionStatus();
            switch (nStatus) {
            case ILG_CONNECTION_DISCONNECTED:
                std::cout << "{!} Конвертер отключён" << std::endl;
                break;

            case ILG_CONNECTION_CONNECTED:
                std::cout << "{!} Конвертер подключён" << std::endl;
                break;

            case ILG_CONNECTION_CONNECTING:
                std::cout << "{!} Идёт подключение к конвертеру" << std::endl;
                break;

            default:
                break;
            }
        }
        catch (const std::exception& e) {
            std::cerr << e.what() << std::endl;
        }
    }
}

ilg_port_type GetPortTypeByName(const char* pszPortName) {
    if (*pszPortName == '\0')
        return ILG_PORT_TYPE_UNKNOWN;
    if (strncmp(pszPortName, "/dev/tty", 8) == 0)
        return ILG_PORT_TYPE_COMPORT;
    auto p = strchr(pszPortName, '@');
    if (p != nullptr) {
        int nSn;
        uint nTcpPort;
        if (sscanf(pszPortName, "%d@%u", &nSn, &nTcpPort) == 2)
            return ILG_PORT_TYPE_CLIENT;
        // 30:00000000@192.168.1.98:25001
        return ILG_PORT_TYPE_PROXY;
    }
    uint a[4];
    if (sscanf(pszPortName, "%u.%u.%u.%u", &a[0], &a[1], &a[2], &a[3]) == 4)
        return ILG_PORT_TYPE_SERVER;
    return ILG_PORT_TYPE_UNKNOWN;
}

bool DoConnectToConverter(CILG& oILG, CConverter& oConverter) {
    CConverterSearch oSearch(oILG.GetSearch());
#if 0
    // Устанавливаем TCP-порты конвертеров в режиме "Клиент"
    const uint16_t aListenPorts[] = {25000};
    oSearch.SetListenPorts(aListenPorts, std::size(aListenPorts));
#endif
    // Ищем конвертеры
    std::cout << "Поиск конвертеров..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetConverterCount();
    std::cout << "Найдено конвертеров: " << nCount << std::endl;

    constexpr int kEnterPortNameCommand = 99;
    int nCommand;
    if (nCount > 0) {
        std::cout << std::endl << "Выберите конвертер:" << std::endl;
        for (size_t i = 0; i < nCount; i++) {
            ilg_converter_info rInfo;
            oSearch.GetConverterInfo(i, rInfo);

            std::stringstream ss;
            if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
                ss << kConverterModelNames[rInfo.nModel];
            if (rInfo.nSn != -1)
                ss << " с/н:" << rInfo.nSn;
            if (rInfo.nFwVersion != 0)
                ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
            if (rInfo.nFwBuildDate != 0)
                ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
            std::cout << 1 + i << ". " << rInfo.pszPortName << " [" << rInfo.pszConnect
                      << "]: " << ss.str() << std::endl;
        }
        std::cout << kEnterPortNameCommand << " - Ввести имя порта..." << std::endl;
        std::cout << "0 - Выйти из программы" << std::endl;

        std::cin >> nCommand;
        std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
        if (std::cin.fail()) {
            std::cin.clear();
            nCommand = -1;
        }
    }
    else
        nCommand = kEnterPortNameCommand;
    ilg_port_type nPortType = ILG_PORT_TYPE_UNKNOWN;
    std::string sPortName;
    ilg_converter_model nConnectModel = ILG_CONVERTER_MODEL_UNKNOWN;
    switch (nCommand) {
    case 0:
        return false;

    case kEnterPortNameCommand:
        {
            std::cout << "Введите имя порта конвертера:" << std::endl;
            std::getline(std::cin, sPortName);
            nPortType = GetPortTypeByName(sPortName.c_str());
        }
        break;

    default:
        if ((nCommand > 0) && (static_cast<size_t>(nCommand) <= nCount)) {
            ilg_converter_info rInfo;
            oSearch.GetConverterInfo(static_cast<size_t>(nCommand - 1), rInfo);
            nPortType = rInfo.nPortType;
            sPortName = rInfo.pszPortName;
            nConnectModel = rInfo.nModel;
        }
        break;
    }
    if (nPortType != ILG_PORT_TYPE_UNKNOWN) {
        // Подключаемся к конвертеру
        // Дескриптор поиска oSearch не удаляем пока не подключимся, т.к. в нём удерживаются
        // подключения к конвертерам, подключённым к IP конвертеру в режиме "Клиент"
        std::cout << "Подключение к конвертеру [" << kPortTypeNames[nPortType] << ": " << sPortName
                  << "]..." << std::endl;
        oConverter = oILG.GetConverter(nPortType, sPortName.c_str());
        // Подписываемся на уведомления от конвертера
        oConverter.SetMessageCallback(MessageCallback, &oConverter);
        if (nConnectModel != ILG_CONVERTER_MODEL_UNKNOWN) {
            ilg_converter_options rOptions;
            oConverter.GetOptions(rOptions);
            rOptions.nConnectModel = nConnectModel;
            oConverter.SetOptions(rOptions);
        }
        oConverter.Connect();
        // Получаем информацию о конвертере
        ilg_converter_info rInfo;
        oConverter.GetConverterInfo(rInfo);
        std::stringstream ss;
        if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
            ss << kConverterModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        std::cout << "Конвертер успешно подключён [" << ss.str() << ']' << std::endl;
    }
    return true;
}

int main() {
    try {
#ifdef ILG_LOG
#ifdef ILG_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        // Включаем лог отладки
        CILG::SetLogCallback(LogCallback);
        CILG::SetLogLevel(ILG_LOG_LEVEL_DEBUG);
#endif

        CILG oILG;
        // Настраиваем параметры библиотеки
        {
            ilg_options rOptions;
            oILG.GetOptions(rOptions);
            rOptions.fTwoStopBits = ILG_FALSE;
            rOptions.nRequestTimeout = 3000;
            rOptions.nRequestAttempts = 2;
            oILG.SetOptions(rOptions);
        }
        CConverter oConverter;

        if (!DoConnectToConverter(oILG, oConverter))
            return 0;

        while (true) {
            std::cout << "-----" << std::endl;
            std::cout << "Введите номер команды:" << std::endl;
            std::cout << "1 - Подключиться к..." << std::endl;
            std::cout << "2 - Подключиться" << std::endl;
            std::cout << "3 - Отключиться" << std::endl;
            std::cout << "0 - Выйти из программы" << std::endl;

            int nCommand;
            std::cin >> nCommand;
            if (std::cin.fail()) {
                std::cin.clear();
                nCommand = -1;
            }
            std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');

            switch (nCommand) {
            case 0:
                return 0;

            case 1:
                if (!DoConnectToConverter(oILG, oConverter))
                    return 0;
                break;

            case 2:
                oConverter.Connect();
                break;

            case 3:
                oConverter.Disconnect();
                break;
            }
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
