#include "CConverterDialog.h"

#include <fstream>
#include <iostream>

#include "utils.h"

CConverterDialog::CConverterDialog() :
    m_VBox(Gtk::Orientation::VERTICAL),
    m_StatusFrame("Состояние подключения"),
    m_ConnectButton("Подключиться"),
    m_DisconnectButton("Отключиться"),
    m_SetFwButton("Установить прошивку..."),
    m_CtrBtnsBox(Gtk::Orientation::VERTICAL),
    m_CtrOpenButton("Открыть..."),
    m_oCtrDlg(),
    m_LicBtnsBox(Gtk::Orientation::VERTICAL),
    m_LicReadButton("Читать все"),
    m_LicSetButton("Установить..."),
    m_LicClearButton("Удалить все"),
    m_nConverterSn(-1),
    m_fSettingsModified(false),
    m_fLicensesInit(false) {
    set_destroy_with_parent(true);

    set_title("Конвертер");
    set_default_size(700, 500);

    m_VBox.set_margin(5);
    set_child(m_VBox);
    // m_VBox.set_expand();

    m_oCtrDlg.set_transient_for(*this);
    m_oCtrDlg.set_modal();
    m_oCtrDlg.set_hide_on_close();
    m_oCtrDlg.signal_hide().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_controller_dialog_hide));

    m_TopBox.set_margin(5);
    m_TopBox.set_expand(false);

    m_Notebook.set_margin(10);
    m_Notebook.set_expand();

    m_VBox.append(m_TopBox);
    m_VBox.append(m_Notebook);

    m_StatusFrame.set_margin(10);
    m_StatusFrame.set_expand();
    m_TopBox.append(m_StatusFrame);
    m_StatusBox.set_margin(5);
    m_StatusBox.set_spacing(5);
    m_StatusBox.set_expand();
    m_StatusFrame.set_child(m_StatusBox);

    m_StatusBox.append(m_StatusLabel);
    m_StatusLabel.set_hexpand(true);
    m_StatusLabel.set_halign(Gtk::Align::START);
    m_ConnectButton.set_halign(Gtk::Align::END);
    m_ConnectButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_button_connect));
    m_StatusBox.append(m_ConnectButton);
    m_DisconnectButton.set_halign(Gtk::Align::END);
    m_DisconnectButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_button_disconnect));
    m_StatusBox.append(m_DisconnectButton);
    m_SetFwButton.set_hexpand(true);
    m_SetFwButton.set_halign(Gtk::Align::END);
    m_SetFwButton.set_valign(Gtk::Align::START);
    m_SetFwButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_button_setfw));
    m_TopBox.append(m_SetFwButton);

    m_CtrsBox.set_margin(5);
    m_CtrsBox.set_expand();
    m_LicsBox.set_margin(5);
    m_LicsBox.set_expand();
    m_Notebook.append_page(m_CtrsBox, "Контроллеры");
    m_Notebook.append_page(m_LicsBox, "Лицензии");
    m_Notebook.signal_switch_page().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_notebook_switch_page));

    // Add the ColumnView, inside a ScrolledWindow, with the button underneath:
    m_CtrScrolledWindow.set_child(m_CtrColumnView);
    // Only show the scrollbars when they are necessary:
    m_CtrScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_CtrScrolledWindow.set_expand();

    // Create the List model:
    m_CtrListStore = Gio::ListStore<CtrModelColumns>::create();

    // Set list model and selection model.
    auto selection_model = Gtk::SingleSelection::create(m_CtrListStore);
    selection_model->set_autoselect(false);
    selection_model->set_can_unselect(true);
    m_CtrColumnView.set_model(selection_model);
    m_CtrColumnView.add_css_class("data-table");  // high density table
    selection_model->signal_selection_changed().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_ctr_selection_changed));

    // Add the ColumnView's columns:

    // Столбец "Адрес"
    auto factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_address));
    auto column = Gtk::ColumnViewColumn::create("Адрес", factory);
    m_CtrColumnView.append_column(column);

    // Столбец "Модель"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_model));
    column = Gtk::ColumnViewColumn::create("Модель", factory);
    m_CtrColumnView.append_column(column);

    // Столбец "С/н"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_sn));
    column = Gtk::ColumnViewColumn::create("С/н", factory);
    m_CtrColumnView.append_column(column);

    // Столбец "Версия прошивки"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_fwversion));
    column = Gtk::ColumnViewColumn::create("Версия прошивки", factory);
    m_CtrColumnView.append_column(column);

    // Столбец "Режим считывателя"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_readermode));
    column = Gtk::ColumnViewColumn::create("Режим считывателя", factory);
    m_CtrColumnView.append_column(column);

    m_refGesture = Gtk::GestureClick::create();
    m_refGesture->set_button(GDK_BUTTON_SECONDARY);
    m_refGesture->signal_pressed().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_ctr_columnview_rclick));
    m_CtrColumnView.add_controller(m_refGesture);

    m_CtrColumnView.signal_activate().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_ctr_columnview_activate));

    m_CtrsBox.append(m_CtrScrolledWindow);
    m_CtrsBox.append(m_CtrBtnsBox);
    m_CtrBtnsBox.append(m_CtrOpenButton);
    m_CtrOpenButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_button_ctr_open));

    // Add the ColumnView, inside a ScrolledWindow, with the button underneath:
    m_LicScrolledWindow.set_child(m_LicColumnView);
    // Only show the scrollbars when they are necessary:
    m_LicScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_LicScrolledWindow.set_expand();

    // Create the List model:
    m_LicListStore = Gio::ListStore<LicModelColumns>::create();

    // Set list model and selection model.
    selection_model = Gtk::SingleSelection::create(m_LicListStore);
    selection_model->set_autoselect(false);
    selection_model->set_can_unselect(true);
    m_LicColumnView.set_model(selection_model);
    m_LicColumnView.add_css_class("data-table");  // high density table

    // Add the ColumnView's columns:

    // Столбец "Лицензия"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_license));
    column = Gtk::ColumnViewColumn::create("Лицензия", factory);
    m_LicColumnView.append_column(column);

    // Столбец "Контроллеры"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_controllers));
    column = Gtk::ColumnViewColumn::create("Контроллеры", factory);
    m_LicColumnView.append_column(column);

    // Столбец "Ключи"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_keys));
    column = Gtk::ColumnViewColumn::create("Ключи", factory);
    m_LicColumnView.append_column(column);

    // Столбец "Годно до"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_expdate));
    column = Gtk::ColumnViewColumn::create("Годно до", factory);
    m_LicColumnView.append_column(column);

    // Столбец "Минут"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CConverterDialog::on_bind_minutes));
    column = Gtk::ColumnViewColumn::create("Минут", factory);
    m_LicColumnView.append_column(column);

    m_LicsBox.append(m_LicScrolledWindow);

    m_LicBtnsBox.set_margin(5);
    m_LicBtnsBox.set_spacing(5);
    m_LicsBox.append(m_LicBtnsBox);
    m_LicBtnsBox.set_halign(Gtk::Align::END);

    // m_LicReadButton.set_halign(Gtk::Align::END);
    m_LicReadButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_button_lic_read));
    m_LicBtnsBox.append(m_LicReadButton);
    m_LicSetButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_button_lic_set));
    m_LicBtnsBox.append(m_LicSetButton);
    // m_LicClearButton.set_halign(Gtk::Align::END);
    m_LicClearButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CConverterDialog::on_button_lic_clear));
    m_LicBtnsBox.append(m_LicClearButton);

    m_oDisp.connect(sigc::mem_fun(*this, &CConverterDialog::on_ilg));

    // Create actions:
    auto refActionGroup = Gio::SimpleActionGroup::create();
    m_refAutoScanAction = refActionGroup->add_action_bool(
        "autoscan", sigc::mem_fun(*this, &CConverterDialog::on_menu_ctrs_autoscan), false);
    m_refReassign01Action = refActionGroup->add_action_bool(
        "reassign01", sigc::mem_fun(*this, &CConverterDialog::on_menu_ctrs_reassign01), false);
    m_refNetCtrsAction = refActionGroup->add_action_bool(
        "netctrs", sigc::mem_fun(*this, &CConverterDialog::on_menu_ctrs_netctrs), false);
    m_refELockAction = refActionGroup->add_action_bool(
        "elock", sigc::mem_fun(*this, &CConverterDialog::on_menu_ctrs_elock), false);
    m_refScanMethAction = refActionGroup->add_action_radio_integer(
        "scanmeth", sigc::mem_fun(*this, &CConverterDialog::on_menu_scanmeth), 1);
    refActionGroup->add_action("refresh",
                               sigc::mem_fun(*this, &CConverterDialog::on_menu_ctrs_refresh));

    insert_action_group("ctrs", refActionGroup);

    m_refBuilder = Gtk::Builder::create();
    // Layout the actions in a menubar and toolbar:
    Glib::ustring ui_info =
        "<interface>"
        "  <menu id='menu-ctrs'>"
        "    <section>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Авто поиск контроллеров</attribute>"
        "        <attribute name='action'>ctrs.autoscan</attribute>"
        "      </item>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Переназначать адреса 0 и 1</attribute>"
        "        <attribute name='action'>ctrs.reassign01</attribute>"
        "      </item>"
        "    </section>"
        "    <submenu>"
        "      <attribute name='label' translatable='yes'>Типы устройств</attribute>"
        "      <section>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Сетевые контроллеры</attribute>"
        "        <attribute name='action'>ctrs.netctrs</attribute>"
        "      </item>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Электронные замки</attribute>"
        "        <attribute name='action'>ctrs.elock</attribute>"
        "      </item>"
        "      </section>"
        "    </submenu>"
        "    <submenu>"
        "      <attribute name='label' translatable='yes'>Способ сканирования</attribute>"
        "      <section>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Авто</attribute>"
        "        <attribute name='action'>ctrs.scanmeth</attribute>"
        "        <attribute name='target' type='i'>0</attribute>"
        "      </item>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Сканирование диапазонов</attribute>"
        "        <attribute name='action'>ctrs.scanmeth</attribute>"
        "        <attribute name='target' type='i'>1</attribute>"
        "      </item>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Опрос адресов</attribute>"
        "        <attribute name='action'>ctrs.scanmeth</attribute>"
        "        <attribute name='target' type='i'>2</attribute>"
        "      </item>"
        "      </section>"
        "    </submenu>"
        "    <section>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Обновить</attribute>"
        "        <attribute name='action'>ctrs.refresh</attribute>"
        "      </item>"
        "    </section>"
        "  </menu>"
        "</interface>";

    try {
        m_refBuilder->add_from_string(ui_info);
    }
    catch (const Glib::Error& ex) {
        std::cerr << "building menus failed: " << ex.what();
    }

    // Get the menu:
    auto gmenu = m_refBuilder->get_object<Gio::Menu>("menu-ctrs");
    if (!gmenu)
        g_warning("GMenu not found");

    m_CtrMenuPopup.set_parent(m_CtrScrolledWindow);
    m_CtrMenuPopup.set_menu_model(gmenu);
    m_CtrMenuPopup.set_has_arrow(false);
}

CConverterDialog::~CConverterDialog() {
    m_CtrMenuPopup.unparent();
}

void CConverterDialog::SetConverter(ilg::CConverter& oConverter) {
    if (oConverter) {
        m_oConverter = std::move(oConverter);
        UpdateConnectionStatus();
        UpdateConverterInfo();
        LoadSettings();
        ilg_converter_options rOptions;
        m_oConverter.GetOptions(rOptions);
        m_refReassign01Action->change_state(static_cast<bool>(rOptions.fReassignAddress01));
        m_refNetCtrsAction->change_state((rOptions.nScanDeviceTypes & ILG_CTR_TYPE_F_NETCTR) != 0);
        m_refELockAction->change_state((rOptions.nScanDeviceTypes & ILG_CTR_TYPE_F_ELOCK) != 0);
        m_refScanMethAction->change_state((int)rOptions.nScanMethod);
        m_refAutoScanAction->change_state(true);
        m_oConverter.SetAutoScan();
        m_oConverter.EnableMessageQueue();
        m_oConverter.SetMessageCallback(&on_ilg_message, this);
        UpdateControllers();
    }
    else {
        m_oConverter.SetMessageCallback(nullptr);
        if (m_fSettingsModified)
            SaveSettings();
        m_oConverter = std::move(oConverter);
    }
}

bool CConverterDialog::DoCtrOpen() {
    auto pModel = dynamic_cast<Gtk::SingleSelection*>(m_CtrColumnView.get_model().get());
    auto item = std::dynamic_pointer_cast<CtrModelColumns>(pModel->get_selected_item());
    if (nullptr == item)
        return false;

    auto oController = m_oConverter.GetController(item->m_nModel, item->m_nSn);
    try {
        oController.Connect();
    }
    catch (const std::exception& e) {
        ShowMessage(e.what(), Gtk::MessageType::ERROR);
        return false;
    }

    m_oCtrDlg.SetController(oController);
    m_oCtrDlg.set_visible(true);
    return true;
}

void ILG_CALL CConverterDialog::on_ilg_message(ilg_converter_msg nMsg, const void* pMsgData,
                                               void* pUserData) {
    auto p = (CConverterDialog*)pUserData;
    p->m_oDisp.emit();
}

void CConverterDialog::on_ilg() {
    ilg_converter_msg nMsg;
    const void* pMsgData;
    while (m_oConverter.GetMessage(nMsg, pMsgData)) {
        switch (nMsg) {
        case ILG_CONVERTER_MSG_COMMAND_FINISH:
            if (m_refProgressDialog != nullptr) {
                auto nStatus = m_oCommand.GetStatus();
                m_oCommand = nullptr;
                m_refProgressDialog = nullptr;
                if (ILG_FAILED(nStatus))
                    ShowMessage(ilg_get_error_text(nStatus), Gtk::MessageType::ERROR);
            }
            break;

        case ILG_CONVERTER_MSG_CONNECTION_CHANGED:
            UpdateConnectionStatus();
            if (m_oConverter.GetConnectionStatus() == ILG_CONNECTION_CONNECTED)
                UpdateConverterInfo();
            break;

        case ILG_CONVERTER_MSG_CONTROLLER_LIST_CHANGED:
            UpdateControllers();
            break;
        }
    }
}

void CConverterDialog::UpdateConnectionStatus() {
    auto nStatus = m_oConverter.GetConnectionStatus();
    switch (nStatus) {
    case ILG_CONNECTION_CONNECTED:
        m_StatusLabel.set_text("Подключён");
        break;

    case ILG_CONNECTION_DISCONNECTED:
        m_StatusLabel.set_text("Отключён");
        break;

    case ILG_CONNECTION_CONNECTING:
        m_StatusLabel.set_text("Подключение...");
        break;
    }
}

void CConverterDialog::UpdateConverterInfo() {
    ilg_converter_info rInfo;
    m_oConverter.GetConverterInfo(rInfo);
    m_nConverterSn = rInfo.nSn;
    std::stringstream ss;
    ss << "Конвертер (" << rInfo.pszPortName << "): " << ilg::kConverterModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н: " << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << ilg::VersionToStr(rInfo.nFwVersion);
    if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
        ss << " [" << ilg::kConverterModeNames[rInfo.nMode] << ']';
    set_title(ss.str());
}

void CConverterDialog::UpdateControllers() {
    auto nCount = m_oConverter.GetControllerCount();
    m_CtrListStore->remove_all();
    ilg_controller_info rInfo;
    for (size_t i = 0; i < nCount; i++) {
        m_oConverter.GetControllerInfo(i, rInfo);
        m_CtrListStore->append(CtrModelColumns::create(rInfo.nAddress, rInfo.nModel, rInfo.nSn,
                                                       rInfo.nFwVersion,
                                                       rInfo.nCtrFlags & ILG_CTR_F_WIEGAND));
    }
    UpdateCtrButtonsState();
}

void CConverterDialog::UpdateCtrButtonsState() {
    auto pModel = dynamic_cast<Gtk::SingleSelection*>(m_CtrColumnView.get_model().get());
    auto item = std::dynamic_pointer_cast<CtrModelColumns>(pModel->get_selected_item());
    m_CtrOpenButton.set_sensitive(item != nullptr);
}

void CConverterDialog::UpdateLicenses() {
    auto nCount = m_oConverter.GetLicenseCount();
    m_LicListStore->remove_all();
    ilg_license_info rInfo;
    for (size_t i = 0; i < nCount; i++) {
        m_oConverter.GetLicenseInfo(i, rInfo);
        m_LicListStore->append(LicModelColumns::create(
            rInfo.nLicenseN, rInfo.nControllers, rInfo.nKeys, rInfo.nExpireDate, rInfo.nCountdown));
    }
    m_fLicensesInit = true;
}

void CConverterDialog::LoadSettings() {
    m_fSettingsModified = false;
    auto sPath(GetSettingFilePath());
    if (std::filesystem::exists(sPath)) {
        std::ifstream inFile;
        inFile.open(sPath);
        ilg_converter_options rOptions;
        m_oConverter.GetOptions(rOptions);
        inFile >> std::dec >> rOptions.nSpeed;
        bool f;
        inFile >> std::boolalpha >> f;
        rOptions.fNormalMode = static_cast<ilg_bool>(f);
        inFile >> std::dec >> rOptions.nReconnectPeriod >> rOptions.nScanPeriod;
        inFile >> std::hex >> rOptions.nScanDeviceTypes;
        uint n;
        inFile >> std::dec >> n;
        if (n < ILG_CTR_SCAN_METHOD_SIZE)
            rOptions.nScanMethod = static_cast<ilg_ctr_scan_method>(n);
        inFile >> rOptions.nLastPolledAddress >> rOptions.nPollTimeout;
        inFile >> std::boolalpha >> f;
        rOptions.fReassignAddress01 = static_cast<ilg_bool>(f);
        m_oConverter.SetOptions(rOptions);
    }
}

void CConverterDialog::SaveSettings() {
    m_fSettingsModified = false;
    std::filesystem::path sPath(GetSettingFilePath());
    std::ofstream outFile;
    outFile.open(sPath);
    ilg_converter_options rOptions;
    m_oConverter.GetOptions(rOptions);
    outFile << std::dec << rOptions.nSpeed;
    outFile << std::boolalpha << static_cast<bool>(rOptions.fNormalMode);
    outFile << std::dec << rOptions.nReconnectPeriod << rOptions.nScanPeriod;
    outFile << std::hex << std::showbase << rOptions.nScanDeviceTypes;
    outFile << std::dec << static_cast<uint>(rOptions.nScanMethod);
    outFile << rOptions.nLastPolledAddress << rOptions.nPollTimeout;
    outFile << std::boolalpha << static_cast<bool>(rOptions.fReassignAddress01);
}

std::filesystem::path CConverterDialog::GetSettingFilePath() const {
    char szCfgPath[PATH_MAX];
    auto count = readlink("/proc/self/exe", szCfgPath, std::size(szCfgPath) - 1);
    szCfgPath[count] = '\0';
    std::stringstream ss;
    ss << szCfgPath << "converter_" << m_nConverterSn << ".txt";
    return ss.str();
}

void CConverterDialog::ShowMessage(const std::string& sMessage, Gtk::MessageType nType) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    m_refDialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_dialog_response)));
}

void CConverterDialog::on_button_connect() {
    m_oConverter.Connect();
}

void CConverterDialog::on_button_disconnect() {
    m_oConverter.Disconnect();
}

void CConverterDialog::on_button_setfw() {
    auto dialog = new Gtk::FileChooserDialog("Выбор файла прошивки конвертера");
    dialog->set_transient_for(*this);
    dialog->set_modal();
    dialog->add_button("Отмена", Gtk::ResponseType::CANCEL);
    dialog->add_button("OK", Gtk::ResponseType::ACCEPT);

    auto fw_filter = Gtk::FileFilter::create();
    fw_filter->add_pattern("*.rom");
    fw_filter->set_name("Прошивки");
    dialog->add_filter(fw_filter);
    dialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_fw_dialog_response), dialog));
    dialog->show();
}

void CConverterDialog::on_fw_dialog_response(int response_id, Gtk::FileChooserDialog* dialog) {
    std::unique_ptr<Gtk::FileChooserDialog> oDialogPtr(dialog);
    if (response_id != Gtk::ResponseType::ACCEPT)
        return;

    auto file = dialog->get_file();
    std::filesystem::path sPath = file->get_path();

    if (std::filesystem::exists(sPath)) {
        std::ifstream file(sPath, std::ios::binary | std::ios::ate);
        auto size = file.tellg();
        file.seekg(0, std::ios::beg);
        std::vector<uint8_t> oData(size);
        file.read((char*)oData.data(), size);

        m_oCommand = m_oConverter.Begin_SetFirmware(oData.data(), oData.size());
        m_refProgressDialog.reset(new CProgressDialog());
        m_refProgressDialog->set_transient_for(*this);
        m_refProgressDialog->set_modal();
        m_refProgressDialog->set_hide_on_close();
        m_refProgressDialog->signal_hide().connect(
            sigc::mem_fun(*this, &CConverterDialog::on_progress_dialog_hide));
        m_refProgressDialog->signal_cancel().connect(
            sigc::mem_fun(*this, &CConverterDialog::on_progress_dialog_cancel));
        m_refProgressDialog->signal_progress().connect(
            sigc::mem_fun(*this, &CConverterDialog::on_progress_dialog_progress));
        m_refProgressDialog->set_visible(true);
    }
}

void CConverterDialog::on_progress_dialog_cancel() {
    m_oCommand.Cancel();
    m_refProgressDialog = nullptr;
    m_oCommand = nullptr;
}

bool CConverterDialog::on_progress_dialog_progress(size_t& nCurrent, size_t& nTotal) {
    m_oCommand.GetProgress(nCurrent, nTotal);
    auto nStatus = m_oCommand.GetStatus();
    if (nStatus != ILG_E_PENDING) {
        m_refProgressDialog = nullptr;
        m_oCommand = nullptr;
        if (ILG_FAILED(nStatus))
            ShowMessage(ilg_get_error_text(nStatus), Gtk::MessageType::ERROR);
        return false;
    }
    return true;
}

void CConverterDialog::on_progress_dialog_hide() {
    m_refProgressDialog = nullptr;
    m_oCommand = nullptr;
}

void CConverterDialog::on_ctr_selection_changed(guint, guint) {
    UpdateCtrButtonsState();
}

void CConverterDialog::on_setup_label(const Glib::RefPtr<Gtk::ListItem>& list_item,
                                      Gtk::Align halign) {
    list_item->set_child(*Gtk::make_managed<Gtk::Label>("", halign));
}

void CConverterDialog::on_bind_address(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CtrModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(Glib::ustring::sprintf("%u", col->m_nAddress));
}

void CConverterDialog::on_bind_model(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CtrModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(ilg::kControllerModelNames[col->m_nModel]);
}

void CConverterDialog::on_bind_sn(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CtrModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(Glib::ustring::sprintf("%d", col->m_nSn));
}

void CConverterDialog::on_bind_fwversion(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CtrModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(ilg::VersionToStr(col->m_nFwVersion));
}

void CConverterDialog::on_bind_readermode(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CtrModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(kReaderMode[col->m_fWiegand]);
}

void CConverterDialog::on_menu_ctrs_autoscan() {
    auto fActive = false;
    m_refAutoScanAction->get_state(fActive);
    fActive = !fActive;
    m_refAutoScanAction->change_state(fActive);
    m_oConverter.SetAutoScan(fActive);
}

void CConverterDialog::on_menu_ctrs_reassign01() {
    auto fActive = false;
    m_refReassign01Action->get_state(fActive);
    fActive = !fActive;
    m_refReassign01Action->change_state(fActive);
    ilg_converter_options rOptions;
    m_oConverter.GetOptions(rOptions);
    rOptions.fReassignAddress01 = static_cast<ilg_bool>(fActive);
    m_oConverter.SetOptions(rOptions);
    m_fSettingsModified = true;
}

void CConverterDialog::on_menu_ctrs_netctrs() {
    auto fActive = false;
    m_refNetCtrsAction->get_state(fActive);
    fActive = !fActive;
    m_refNetCtrsAction->change_state(fActive);
    ilg_converter_options rOptions;
    m_oConverter.GetOptions(rOptions);
    if (fActive)
        rOptions.nScanDeviceTypes |= ILG_CTR_TYPE_F_NETCTR;
    else
        rOptions.nScanDeviceTypes &= ~ILG_CTR_TYPE_F_NETCTR;
    m_oConverter.SetOptions(rOptions);
    m_fSettingsModified = true;
}

void CConverterDialog::on_menu_ctrs_elock() {
    auto fActive = false;
    m_refELockAction->get_state(fActive);
    fActive = !fActive;
    m_refELockAction->change_state(fActive);
    ilg_converter_options rOptions;
    m_oConverter.GetOptions(rOptions);
    if (fActive)
        rOptions.nScanDeviceTypes |= ILG_CTR_TYPE_F_ELOCK;
    else
        rOptions.nScanDeviceTypes &= ~ILG_CTR_TYPE_F_ELOCK;
    m_oConverter.SetOptions(rOptions);
    m_fSettingsModified = true;
}

void CConverterDialog::on_menu_scanmeth(int parameter) {
    m_refScanMethAction->change_state(parameter);

    if ((parameter >= 0) && (parameter < ILG_CTR_SCAN_METHOD_SIZE)) {
        ilg_converter_options rOptions;
        m_oConverter.GetOptions(rOptions);
        rOptions.nScanMethod = (ilg_ctr_scan_method)parameter;
        m_oConverter.SetOptions(rOptions);
        m_fSettingsModified = true;
    }
}

void CConverterDialog::on_menu_ctrs_refresh() {
    m_oConverter.Scan(true);
    UpdateControllers();
}

void CConverterDialog::on_ctr_columnview_rclick(int n_press, double x, double y) {
    const Gdk::Rectangle rect(x, y, 1, 1);
    m_CtrMenuPopup.set_pointing_to(rect);
    m_CtrMenuPopup.popup();
}

void CConverterDialog::on_ctr_columnview_activate(guint n) {
    DoCtrOpen();
}

void CConverterDialog::on_button_ctr_open() {
    DoCtrOpen();
}

void CConverterDialog::on_controller_dialog_hide() {
    ilg::CController oController;
    m_oCtrDlg.SetController(oController);
    m_oCtrDlg.set_visible(false);
}

void CConverterDialog::on_bind_license(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<LicModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(Glib::ustring::sprintf("%u", col->m_nLicenseN));
}

void CConverterDialog::on_bind_controllers(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<LicModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_nControllers != 0xff)
        label->set_text(Glib::ustring::sprintf("%u", col->m_nControllers));
    else
        label->set_text("∞");
}

void CConverterDialog::on_bind_keys(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<LicModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_nKeys != 0xffff)
        label->set_text(Glib::ustring::sprintf("%u", col->m_nKeys));
    else
        label->set_text("∞");
}

void CConverterDialog::on_bind_expdate(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<LicModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_nExpDate != 0xffff) {
        time_t tNow = time(nullptr);
        struct tm* pTemp = localtime(&tNow);
        if (nullptr == pTemp)
            return;
        label->set_text(Glib::ustring::sprintf(
            "%.2u.%.2u.%.4u", (uint)ILG_EXPIREDATE_DAY(col->m_nExpDate),
            (uint)ILG_EXPIDEDATE_MONTH(col->m_nExpDate),
            (uint)ILG_EXPIDEDATE_YEAR(col->m_nExpDate, 1900 + pTemp->tm_year)));
    }
    else
        label->set_text("∞");
}

void CConverterDialog::on_bind_minutes(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<LicModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_nMinutes != 0xffff)
        label->set_text(Glib::ustring::sprintf("%u", col->m_nMinutes));
    else
        label->set_text("∞");
}

void CConverterDialog::on_button_lic_read() {
    UpdateLicenses();
}

void CConverterDialog::on_button_lic_set() {
    auto dialog = new Gtk::FileChooserDialog("Выбор файла лицензии конвертера");
    dialog->set_transient_for(*this);
    dialog->set_modal();
    dialog->add_button("Отмена", Gtk::ResponseType::CANCEL);
    dialog->add_button("OK", Gtk::ResponseType::ACCEPT);

    auto license_filter = Gtk::FileFilter::create();
    license_filter->add_pattern("*.lic");
    license_filter->set_name("Лицензии");
    dialog->add_filter(license_filter);
    dialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CConverterDialog::on_lic_dialog_response), dialog));
    dialog->show();
}

void CConverterDialog::on_button_lic_clear() {
    m_oConverter.ClearLicenses();
    UpdateLicenses();
}

void CConverterDialog::on_lic_dialog_response(int response_id, Gtk::FileChooserDialog* dialog) {
    std::unique_ptr<Gtk::FileChooserDialog> oDialogPtr(dialog);
    if (response_id != Gtk::ResponseType::ACCEPT)
        return;

    auto file = dialog->get_file();
    std::filesystem::path sPath = file->get_path();

    if (std::filesystem::exists(sPath)) {
        uint32_t nLicenseN = ILG_LICENSE_N;
        std::vector<uint8_t> oData;
        std::ifstream inFile(sPath);
        std::string sLine;
        const char* pName;
        const char* pValue;
        while (std::getline(inFile, sLine)) {
            if (sLine.empty() || (sLine.front() == '['))
                continue;
            if (sLine.back() == '\r')
                sLine.pop_back();
            auto nPos = sLine.find('=');
            if (nPos != sLine.npos) {
                sLine[nPos] = '\0';
                pName = sLine.c_str();
                pValue = pName + nPos + 1;
                if (strcmp(pName, "LicName") == 0) {
                    if (strcmp(pValue, "SDKG") == 0)
                        nLicenseN = 5;
                    else if (strcmp(pValue, "Proffit") == 0)
                        nLicenseN = 8;
                    else if (strcmp(pValue, "LITE") == 0)
                        nLicenseN = 6;
                }
                else if (strcmp(pName, "TXT") == 0) {
                    oData = decodeHex(pValue);
                }
            }
        }
        if (!oData.empty()) {
            m_oConverter.WriteLicense(nLicenseN, oData.data(), oData.size());
            UpdateLicenses();
        }
    }
}

void CConverterDialog::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void CConverterDialog::on_notebook_switch_page(Gtk::Widget* page, guint page_num) {
    if ((1 == page_num) && !m_fLicensesInit)
        UpdateLicenses();
}
