#include "ilg_cpp_helpers.h"

#include <time.h>

#include <memory>
#include <sstream>

namespace ilg {

CILGException::CILGException(ilg_status nCode) :
    m_nCode(nCode) {
}

const char* CILGException::what() const throw() {
    return ilg_get_error_text(m_nCode);
}

void ILGCheck(ilg_status nCode) {
    if (ILG_FAILED(nCode))
        throw CILGException(nCode);
}

CKeyNumber::CKeyNumber() {
    ilg_key_number::nData = 0;
}

CKeyNumber::CKeyNumber(const ilg_key_number& other) {
    ilg_key_number::nData = other.nData;
}

CKeyNumber::CKeyNumber(uint8_t nEmSeries, uint16_t nEmNumber, uint32_t nFacility) {
    ilg_key_number::em_marine.nSeries = nEmSeries;
    ilg_key_number::em_marine.nNumber = nEmNumber;
    memcpy(ilg_key_number::em_marine.aFacility, &nFacility,
           std::size(ilg_key_number::em_marine.aFacility));
}

CKeyNumber::CKeyNumber(const uint8_t* pData, size_t nSize) {
    Assign(pData, nSize);
}

void CKeyNumber::Clear() {
    ilg_key_number::nData = 0;
}

void CKeyNumber::Assign(const uint8_t* pData, size_t nSize) {
    ilg_key_number::nData = 0;
    if (nSize != 0) {
        if (nSize > std::size(ilg_key_number::aDallas))
            nSize = std::size(ilg_key_number::aDallas);
        memcpy(ilg_key_number::aDallas, pData, nSize);
    }
}

CILGHandle::CILGHandle() :
    m_h(nullptr) {
}

CILGHandle::CILGHandle(ilg_handle h) :
    m_h(h) {
}

CILGHandle::CILGHandle(CILGHandle&& other) {
    m_h = other.m_h;
    other.m_h = nullptr;
}

CILGHandle::~CILGHandle() {
    if (m_h != nullptr)
        ilg_close_handle(m_h);
}

CILGHandle& CILGHandle::operator=(CILGHandle&& other) {
    if (m_h != nullptr)
        ilg_close_handle(m_h);
    m_h = other.m_h;
    other.m_h = nullptr;
    return *this;
}

CILGHandle::operator ilg_handle() const {
    return m_h;
}

CILGHandle::operator bool() const {
    return m_h != nullptr;
}

void CILGHandle::Swap(CILGHandle& other) noexcept {
    std::swap(m_h, other.m_h);
}

ilg_handle CILGHandle::Get() const {
    return m_h;
}

void CILGHandle::Close() {
    if (m_h != nullptr) {
        ILGCheck(ilg_close_handle(m_h));
        m_h = nullptr;
    }
}

void CILGHandle::Attach(ilg_handle h) {
    if (m_h != nullptr)
        ILGCheck(ilg_close_handle(m_h));
    m_h = h;
}

ilg_handle CILGHandle::Detach() {
    ilg_handle res = m_h;
    m_h = nullptr;
    return res;
}

CAsyncCommand::CAsyncCommand() {
}

CAsyncCommand::CAsyncCommand(ilg_handle h) :
    CILGHandle(h) {
}

CAsyncCommand::CAsyncCommand(CAsyncCommand&& other) :
    CILGHandle(other.m_h) {
    other.m_h = nullptr;
}

CAsyncCommand::~CAsyncCommand() {
}

CAsyncCommand& CAsyncCommand::operator=(CAsyncCommand&& other) {
    Attach(other.Detach());
    return *this;
}

CConverterSearch::CConverterSearch() {
}

CConverterSearch::CConverterSearch(ilg_handle h) :
    CILGHandle(h) {
}

CConverterSearch::CConverterSearch(CConverterSearch&& other) :
    CILGHandle(other.m_h) {
    other.m_h = nullptr;
}

CConverterSearch::~CConverterSearch() {
}

CConverterSearch& CConverterSearch::operator=(CConverterSearch&& other) {
    Attach(other.Detach());
    return *this;
}

void CConverterSearch::GetListenPorts(std::vector<unsigned short>& oPorts) const {
    size_t nSize = ILG_AUTOALLOCATE;
    unsigned short* p = nullptr;
    ILGCheck(ilg_search_get_listen_ports(m_h, (unsigned short*)&p, &nSize));

    std::unique_ptr<void, decltype(ilg_free_memory)*> o(p, ilg_free_memory);
    oPorts.assign(p, p + nSize);
}

CController::CController() {
}

CController::CController(ilg_handle h) :
    CILGHandle(h) {
}

CController::CController(CController&& other) {
    m_h = other.m_h;
    other.m_h = nullptr;
}

CController::~CController() {
}

CController& CController::operator=(CController&& other) {
    Attach(other.Detach());
    return *this;
}

std::string CController::ReadLines() {
    char* pBuf = nullptr;
    size_t nSize = ILG_AUTOALLOCATE;
    ILGCheck(ilg_controller_read_lines(m_h, (char*)&pBuf, &nSize));
    std::unique_ptr<void, decltype(ilg_free_memory)*> o(pBuf, ilg_free_memory);
    return std::string(pBuf, nSize - 1);
}

void CController::End_ReadLines(ilg_handle hCommand, std::string& sLines) {
    const char* pLines = nullptr;
    ILGCheck(ilg_controller_end_read_lines(hCommand, &pLines));
    sLines.assign(pLines);
}

void CController::ReadConfiguration(std::vector<uint8_t>& oData) {
    uint8_t* pBuf = nullptr;
    size_t nSize = ILG_AUTOALLOCATE;
    ILGCheck(ilg_controller_read_configuration(m_h, (uint8_t*)&pBuf, &nSize));
    std::unique_ptr<void, decltype(ilg_free_memory)*> o(pBuf, ilg_free_memory);
    oData.assign(pBuf, pBuf + nSize);
}

void CController::End_ReadConfiguration(ilg_handle hCommand, std::vector<uint8_t>& oData) {
    const uint8_t* pData = nullptr;
    size_t nSize = 0;
    ILGCheck(ilg_controller_end_read_configuration(hCommand, &pData, &nSize));
    oData.assign(pData, pData + nSize);
}

CConverter::CConverter() {
}

CConverter::CConverter(ilg_handle h) :
    CILGHandle(h) {
}

CConverter::CConverter(CConverter&& other) {
    m_h = other.m_h;
    other.m_h = nullptr;
}

CConverter::~CConverter() {
}

CConverter& CConverter::operator=(CConverter&& other) {
    Attach(other.Detach());
    return *this;
}

CILG::CILG(bool fInit) {
    if (fInit) {
        ILGCheck(ilg_init());
        m_fInit = true;
    }
}

void CILG::Init() {
    if (m_fInit)
        return;
    ILGCheck(ilg_init());
    m_fInit = true;
}

CILG::~CILG() {
    if (m_fInit)
        ilg_cleanup();
}

const char* kPortTypeNames[] = {
    "",        // 1
    "COM",     // 2
    "Server",  // 3
    "Client",  // 4
    "Proxy"    // 5
};
static_assert(ILG_PORT_TYPE_SIZE == 5);

const char* kConverterModelNames[] = {
    "",                   // 1
    "Z-397",              // 2
    "Z-397 Strag",        // 3
    "Z-397 Strag 115",    // 4
    "Z-397 Guard",        // 5
    "Z-397 IP",           // 6
    "Z-397 Web",          // 7
    "Z-5R Web",           // 8
    "Matrix II Wi-Fi",    // 9
    "Matrix VI Wi-Fi",    // 10
    "Z-5R Web BT",        // 11
    "Z-5R Wi-Fi",         // 12
    "Matrix-II EH Web",   // 13
    "Matrix-VI EH Web",   // 14
    "Z-5R Web mini",      // 15
    "Matrix VI NFC WiFi"  // 16
};
static_assert(ILG_CONVERTER_MODEL_SIZE == 16);

const char* kControllerModelNames[] = {
    "",                       // 1
    "Gate 2000",              // 2
    "Matrix-II (E K Net)",    // 3
    "Z-5R (Net)",             // 4
    "Z-5R Net 8k",            // 5
    "Z-5R (Net 16000)",       // 6
    "Matrix-III (MF K Net)",  // 7
    "Guard (Net)",            // 8
    "Z-5R (Web)",             // 9
    "Matrix-II (E K Wi-Fi)",  // 10
    "Z-Eurolock",             // 11
    "Z-9 EHT net v2",         // 12
    "Matrix-VI (NFC K Net)",  // 13
    "Matrix-VI (EH K Net)",   // 14
    "Matrix VI Wi-Fi",        // 15
    "Z-5R Web BT",            // 16
    "Z-5R Wi-Fi",             // 17
    "Matrix-II EH Web",       // 18
    "Matrix-VI EH Web",       // 19
    "Z-5R Web mini",          // 20
    "Matrix VI NFC WiFi",     // 21
    "CP Z-2 EH K Relay",      // 22
    "CP Z-2 EH K"             // 23
};
static_assert(ILG_CONTROLLER_MODEL_SIZE == 23);

const char* kConverterModeNames[] = {
    "",          // 1
    "Normal",    // 2
    "Advanced",  // 3
    "Test",      // 4
    "Accept"     // 5
};
static_assert(ILG_CONVERTER_MODE_SIZE == 5);

std::string SdkVersionToStr(uint nVersion) {
    std::ostringstream s;
    if (nVersion != 0) {
        s << ((nVersion >> 24) & 0xff) << "." << ((nVersion >> 16) & 0xff) << "."
          << ((nVersion >> 8) & 0xff);

        uint nType = (nVersion >> 6) & 3;
        if (nType != 3) {
            s << "-";
            switch (nType) {
            case 0:
                s << "alpha";
                break;

            case 1:
                s << "beta";
                break;

            case 2:
                s << "rc";
                break;
            }
            uint revision = (nVersion & 0x3f);
            if (revision != 0)
                s << "." << revision;
        }
    }
    return s.str();
}

std::string VersionToStr(uint nVersion) {
    std::ostringstream s;
    if (0 == nVersion)
        return s.str();  // Версия не известна

    s << ((nVersion >> 24) & 0xff) << "." << ((nVersion >> 16) & 0xff);
    if (nVersion & 0xffff) {
        s << "." << ((nVersion >> 8) & 0xff);
        if (nVersion & 0xff)
            s << "." << (nVersion & 0xff);
    }

    return s.str();
}

std::string TimeToStr(const int64_t& tTime) {
    struct tm* pTemp = localtime(&tTime);
    if (nullptr == pTemp)
        return std::string();
    char szBuffer[128];
    size_t c = strftime(szBuffer, sizeof(szBuffer), "%c", pTemp);
    return std::string(szBuffer, c);
}

std::string KeyNumberToStr(const ilg_key_number& rNumber, uint32_t nKeyFlags, bool fWiegand) {
    std::string res;
    if (nKeyFlags & ILG_KEY_F_DUAL) {
        auto s = KeybCodeToStr(rNumber, 1);
        char szBuf[32];
        auto c = snprintf(szBuf, std::size(szBuf), "(%.3u,%.5u)+<%s>", rNumber.em_marine.nSeries,
                          rNumber.em_marine.nNumber, s.c_str());
        res.assign(szBuf, szBuf + c);
    }
    else if (fWiegand) {
        char szBuf[32];
        int c;
        if (nKeyFlags & ILG_KEY_F_SHORT)
            c = snprintf(szBuf, std::size(szBuf), "%.3u,%.5u", rNumber.em_marine.nSeries,
                         rNumber.em_marine.nNumber);
        else
            c = snprintf(szBuf, std::size(szBuf), "[%.2X%.2X%.2X] %.3u,%.5u",
                         rNumber.em_marine.aFacility[2], rNumber.em_marine.aFacility[1],
                         rNumber.em_marine.aFacility[0], rNumber.em_marine.nSeries,
                         rNumber.em_marine.nNumber);
        res.assign(szBuf, szBuf + c);
    }
    else {
        char szBuf[32];
        int c;
        if (nKeyFlags & ILG_KEY_F_SHORT)
            c = snprintf(szBuf, std::size(szBuf), "%.2X %.2X %.2X", rNumber.aDallas[2],
                         rNumber.aDallas[1], rNumber.aDallas[0]);
        else
            c = snprintf(szBuf, std::size(szBuf), "%.2X %.2X %.2X %.2X %.2X %.2X",
                         rNumber.aDallas[5], rNumber.aDallas[4], rNumber.aDallas[3],
                         rNumber.aDallas[2], rNumber.aDallas[1], rNumber.aDallas[0]);
        res.assign(szBuf, szBuf + c);
    }
    return res;
}

bool TryStrToKeyNum(const char* pStr, ilg_key_number* pNumber, uint8_t* pKeyFlags) {
    bool fRes = false;
    uint nEmSeries, nEmNumber;
    char szBuf[32];
    if (sscanf(pStr, "(%u,%u)+<%[^>]12s>", &nEmSeries, &nEmNumber, szBuf) == 3) {
        pNumber->em_marine.nSeries = static_cast<uint8_t>(nEmSeries);
        pNumber->em_marine.nNumber = static_cast<uint16_t>(nEmNumber);
        ilg_key_number rNumber2;
        fRes = TryParseKeybCodeStr(szBuf, &rNumber2);
        if (fRes) {
            for (size_t i = 3; i < 6; ++i)
                pNumber->aDallas[i] = rNumber2.aDallas[i - 3];
        }
        *pKeyFlags = ILG_KEY_F_DUAL;
    }
    else if (strchr(pStr, ',') != nullptr) {
        if ('[' == *pStr) {
            uint nFacility;
            fRes = sscanf(pStr, "[%x] %u,%u", &nFacility, &nEmSeries, &nEmNumber) == 3;
            if (fRes) {
                pNumber->em_marine.nSeries = static_cast<uint8_t>(nEmSeries);
                pNumber->em_marine.nNumber = static_cast<uint16_t>(nEmNumber);
                memcpy(pNumber->em_marine.aFacility, &nFacility,
                       sizeof(pNumber->em_marine.aFacility));
            }
        }
        else {
            fRes = sscanf(pStr, "%u,%u", &nEmSeries, &nEmNumber) == 2;
            if (fRes) {
                pNumber->em_marine.nSeries = static_cast<uint8_t>(nEmSeries);
                pNumber->em_marine.nNumber = static_cast<uint16_t>(nEmNumber);
                *pKeyFlags = ILG_KEY_F_SHORT;
            }
        }
    }
    else {
        uint a[6] = {0};
        auto c = sscanf(pStr, "%x %x %x %x %x %x", &a[0], &a[1], &a[2], &a[3], &a[4], &a[5]);
        fRes = (c > 0);
        if (fRes) {
            size_t nLen = (c > 6) ? 6 : static_cast<size_t>(c);
            for (size_t i = 0; i < nLen; ++i)
                pNumber->aDallas[nLen - i - 1] = static_cast<uint8_t>(a[i]);
            if (c <= 3)
                *pKeyFlags = ILG_KEY_F_SHORT;
        }
    }
    return fRes;
}

std::string KeybCodeToStr(const ilg_key_number& rNumber, int nDualIdx, char chError) {
    std::string res;
    size_t nSize;
    uint64_t n = 0;
    switch (nDualIdx) {
    case 0:
        nSize = 3;
        memcpy(&n, &rNumber.dual_dallas.aDallas1, 3);
        break;

    case 1:
        nSize = 3;
        memcpy(&n, &rNumber.dual_dallas.aDallas2, 3);
        break;

    default:
        nSize = 6;
        memcpy(&n, &rNumber.nData, nSize);
        break;
    }
    res.resize(nSize * 2);
    uint8_t b;
    for (auto it = res.rbegin(); it != res.rend(); it++) {
        b = (n & 0xF);
        if ((b >= 1) && (b <= 10))
            *it = static_cast<char>('0' + (b % 10));
        else
            *it = chError;
        n >>= 4;
    }
    return res;
}

bool TryParseKeybCodeStr(const char* pStr, ilg_key_number* pNumber) {
    pNumber->nData = 0;
    if (*pStr == '\0')
        return true;
    auto pBuf = pNumber->aDallas;
    auto pBufEnd = pNumber->aDallas + std::size(pNumber->aDallas);
    auto p = pStr + strlen(pStr) - 1;
    auto pEnd = pStr - 1;
    uint8_t b;
    while (p != pEnd) {
        if (!isdigit(*p))
            return false;
        b = static_cast<uint8_t>(*p - '0');
        if (0 == b)
            b = 10;
        *pBuf = b;

        if (--p == pEnd)
            break;
        if (!isdigit(*p))
            return false;
        b = static_cast<uint8_t>(*p - '0');
        if (0 == b)
            b = 10;
        *pBuf |= (b << 4);
        if (++pBuf == pBufEnd)
            break;
        --p;
    }
    return true;
}

};  // namespace ilg
