#include <codecvt>    // для std::wstring_convert
#include <exception>  // для std::exception
#include <fstream>    // для std::fstream
#include <iomanip>    // для std::put_time
#include <iostream>   // для std::cout и std::cin
#include <limits>     // для std::numeric_limits
#include <string>     // для std::string

#include "ilg_cpp_helpers.h"
#include "ilguard/ilguard.h"

// #define ILG_LOG  // Раскомментируйте, чтобы включить показ отладочных сообщений
#define ILG_LOG_FILE  // Писать лог в файл

using namespace ilg;

// Глобальные переменные:
uint8_t g_nBankCount = 0;  // Количество банков ключей контроллера
uint32_t g_nCtrFlags = 0;  // Флаги контроллера

// Названия режимов контроллера
const char* kModeNames[] = {"Неактивный", "Норма", "Блок", "Свободно", "Ожидание"};

#ifdef ILG_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilguard.log";  // Путь к лог файлу

void ILG_CALL LogCallback(ilg_log_level level, const char* pContext, const char* pMessage, void*) {
#if 1  // Запись в файл
    std::fstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

// Показывает текущее расписание контроллера
void ShowSchedule(CController& oController) {
    std::cout << "Временные зоны для прохода:" << std::endl;
    const char32_t szDowTitles[] = U"ПВСЧПСВ";
    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
    char32_t szDows[8] = {0};
    ilg_time_zone aTZs[7];
    ilg_time_zone* pTz;
    auto tStartTime = now();
    for (uint8_t i = 0; i < g_nBankCount; i++) {
        oController.ReadTimeZones(i, 0, aTZs, std::size(aTZs));
        std::cout << "Банк " << i << ":" << std::endl;
        for (size_t j = 0; j < std::size(aTZs); j++) {
            pTz = &aTZs[j];

            for (size_t k = 0; k < 7; k++)
                szDows[k] = (GET_BIT(pTz->nDaysOfWeek, k) ? szDowTitles[k] : '-');

            std::cout << j << "." << cv.to_bytes(szDows) << " " << std::setfill('0') << std::setw(2)
                      << (uint)pTz->nBeginHour << ":" << std::setw(2) << (uint)pTz->nBeginMinute
                      << " " << std::setw(2) << (uint)pTz->nEndHour << ":" << std::setw(2)
                      << (uint)pTz->nEndMinute << std::endl;
        }
        std::cout << std::endl;
    }

    if (g_nCtrFlags & ILG_CTR_F_MODES) {
        std::cout << "Временные зоны для переключения режима контроллера:" << std::endl;
        ilg_mode_time_zone aModeTZs[2];
        ilg_mode_time_zone* pModeTz;
        oController.ReadModeTimeZones(0, aModeTZs, std::size(aModeTZs));
        for (size_t j = 0; j < std::size(aModeTZs); j++) {
            pModeTz = &aModeTZs[j];

            for (size_t k = 0; k < 7; k++)
                szDows[k] = (GET_BIT(pTz->nDaysOfWeek, k) ? szDowTitles[k] : '-');

            std::cout << j << "." << cv.to_bytes(szDows) << " " << std::setfill('0') << std::setw(2)
                      << (uint)pTz->nBeginHour << ":" << std::setw(2) << (uint)pTz->nBeginMinute
                      << " " << std::setw(2) << (uint)pTz->nEndHour << ":" << std::setw(2)
                      << (uint)pTz->nEndMinute << " " << kModeNames[pModeTz->nMode] << std::endl;
        }
    }
    std::cout << "Выполнено за " << since(tStartTime).count() << " мс" << std::endl;
}

// Устанавливает параметры одной временной зоны контроллера
void SetTimezone(CController& oController) {
    std::cout << "Введите номер банка, номер временной зоны, дни недели (ПВСЧПСВ), время начала "
                 "(ЧЧ:ММ), время конца (ЧЧ:ММ):"
              << std::endl;
    uint nBankN, nZoneN;
    std::string sDows;
    std::tm tmBegin = {};
    std::tm tmEnd = {};
    std::cin >> nBankN >> nZoneN >> sDows >> std::get_time(&tmBegin, "%H:%M") >>
        std::get_time(&tmEnd, "%H:%M");
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
    if (std::cin.fail()) {
        std::cout << "Неправильный ввод" << std::endl;
        return;
    }

    uint nDows = 0;
    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
    auto s = cv.from_bytes(sDows);
    auto c = std::min(s.length(), 7lu);
    for (size_t i = 0; i < c; i++)
        if (s[i] != '-')
            SET_BIT(nDows, i, true);

    ilg_time_zone rTz = {};
    rTz.nDaysOfWeek = static_cast<uint8_t>(nDows);
    rTz.nBeginHour = static_cast<uint8_t>(tmBegin.tm_hour);
    rTz.nBeginMinute = static_cast<uint8_t>(tmBegin.tm_min);
    rTz.nEndHour = static_cast<uint8_t>(tmEnd.tm_hour);
    rTz.nEndMinute = static_cast<uint8_t>(tmEnd.tm_min);

    std::cout << "Запись... " << std::endl;
    auto tStartTime = now();
    oController.WriteTimeZones(static_cast<uint8_t>(nBankN), static_cast<size_t>(nZoneN), &rTz, 1);
    std::cout << "Записано за " << since(tStartTime).count() << " мс" << std::endl;
}

// Показывает текущие дату и время контроллера
void ShowClock(CController& oController) {
    ilg_rtc_params rParams;
    oController.ReadRtcParams(rParams);
    if (rParams.nDiffSec != std::numeric_limits<int64_t>::max()) {
        time_t t = time(nullptr) + rParams.nDiffSec;
        auto tm = *std::localtime(&t);
        std::cout << "Дата и время: " << std::put_time(&tm, "%d-%m-%Y %H:%M:%S") << std::endl;
    }
    else
        std::cout << "Дата и время не корректные" << std::endl;
}

// Синхронизирует часы контроллера с часами ПК
void SyncClock(CController& oController) {
    auto tStartTime = now();
    oController.SyncClock();
    std::cout << "Синхронизировано за " << since(tStartTime).count() << " мс" << std::endl;
    ShowClock(oController);
}

// Подключается к конвертеру
bool DoConnectToConverter(CILG& oILR, CConverter& oConverter) {
    // Ищем конвертеры
    CConverterSearch oSearch(oILR.GetSearch());
    std::cout << "Поиск конвертеров..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetConverterCount();
    if (0 == nCount) {
        std::cout << "Конвертер не найден" << std::endl;
        return false;
    }
    std::cout << "Найдено конвертеров: " << nCount << std::endl;

    std::cout << std::endl << "Выберите конвертер:" << std::endl;
    for (size_t i = 0; i < nCount; i++) {
        ilg_converter_info rInfo;
        oSearch.GetConverterInfo(i, rInfo);

        std::stringstream ss;
        if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
            ss << kConverterModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
            ss << " режим: " << kConverterModeNames[rInfo.nMode];
        std::cout << 1 + i << "." << rInfo.pszPortName << "[" << rInfo.pszConnect
                  << "]: " << ss.str() << std::endl;
    }
    std::cout << "0 - Выйти из программы" << std::endl;

    int nCommand;
    std::cin >> nCommand;
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
    if (std::cin.fail()) {
        std::cin.clear();
        nCommand = -1;
    }

    if ((nCommand <= 0) || (static_cast<size_t>(nCommand) > nCount))
        return false;

    ilg_converter_info rInfo;
    oSearch.GetConverterInfo(static_cast<size_t>(nCommand - 1), rInfo);

    std::cout << "Подключение к конвертеру [" << kPortTypeNames[rInfo.nPortType] << ": "
              << rInfo.pszPortName << "..." << std::endl;
    oConverter = oILR.GetConverter(rInfo.nPortType, rInfo.pszPortName);
    ilg_converter_options rOptions;
    oConverter.GetOptions(rOptions);
    rOptions.nConnectModel = rInfo.nModel;
    oConverter.SetOptions(rOptions);
    // Подключаемся к конвертеру
    oConverter.Connect();
    // Получаем информацию о конвертере
    oConverter.GetConverterInfo(rInfo);
    std::stringstream ss;
    if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
        ss << kConverterModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н:" << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
    if (rInfo.nFwBuildDate != 0)
        ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
    if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
        ss << " режим: " << kConverterModeNames[rInfo.nMode];
    std::cout << "Конвертер успешно подключён [" << ss.str() << ']' << std::endl;
    return true;
}

// Подключается к контроллеру
bool DoConnectToController(CConverter& oConverter, CController& oController) {
    // Поиск контроллеров
    int nCommand;
    while (true) {
        std::cout << "Выберите контроллер:" << std::endl;
        oConverter.Scan();
        auto nCount = oConverter.GetControllerCount();
        for (size_t i = 0; i < nCount; i++) {
            ilg_controller_info rInfo;
            oConverter.GetControllerInfo(i, rInfo);
            std::stringstream ss;
            if (rInfo.nModel != ILG_CONTROLLER_MODEL_UNKNOWN)
                ss << " " << kControllerModelNames[rInfo.nModel];
            if (rInfo.nSn != -1)
                ss << " с/н:" << rInfo.nSn;
            if (rInfo.nFwVersion != 0)
                ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
            std::cout << i + 1 << ". " << (uint)rInfo.nAddress << " " << ss.str() << std::endl;
        }
        if (0 == nCount)
            std::cout << "1 - Искать снова" << std::endl;
        std::cout << "0 - Выйти из программы" << std::endl;

        std::cin >> nCommand;
        std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
        if (std::cin.fail()) {
            std::cin.clear();
            nCommand = -1;
        }

        if (0 == nCommand)
            return false;
        if ((nCommand >= 1) && (static_cast<size_t>(nCommand) <= nCount))
            break;
    }
    ilg_controller_info rInfo;
    oConverter.GetControllerInfo(static_cast<size_t>(nCommand - 1), rInfo);
    oController = oConverter.GetController(rInfo.nModel, rInfo.nSn);
    // Подключаемся к контроллеру
    std::cout << "Подключение к контроллеру [" << kControllerModelNames[rInfo.nModel] << ": "
              << rInfo.nSn << "]..." << std::endl;
    oController.Connect();
    // Получаем информацию о контроллере
    oController.GetControllerInfo(rInfo);
    std::cout << "Контроллер успешно подключён [#" << rInfo.nAddress << ' '
              << kControllerModelNames[rInfo.nModel] << " с/н:" << rInfo.nSn
              << " прошивка:" << VersionToStr(rInfo.nFwVersion) << ']' << std::endl;
    // Выключаем авто поиск контроллеров (не обязательно)
    oConverter.SetAutoScan(false);
    return true;
}

int main() {
    try {
#ifdef ILG_LOG
#ifdef ILG_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        // Включаем лог отладки
        CILG::SetLogCallback(LogCallback);
        CILG::SetLogLevel(ILG_LOG_LEVEL_DEBUG);
#endif

        CILG oILG;

        // Подключаемся к конвертеру
        CConverter oConverter;
        if (!DoConnectToConverter(oILG, oConverter))
            return 0;

        // Подключаемся к контроллеру
        CController oController;
        if (!DoConnectToController(oConverter, oController))
            return 0;

        ilg_controller_info rInfo;
        oController.GetControllerInfo(rInfo);
        g_nBankCount = (rInfo.nCtrFlags & (ILG_CTR_F_TWOBANKS | ILG_CTR_F_DUALZONE)) ? 2 : 1;
        g_nCtrFlags = rInfo.nCtrFlags;

        while (true) {
            std::cout << "-----" << std::endl;
            std::cout << "Введите номер команды:" << std::endl;
            std::cout << "1 - Показать расписание" << std::endl;
            std::cout << "2 - Установить временную зону..." << std::endl;
            std::cout << "3 - Показать часы" << std::endl;
            std::cout << "4 - Синхронизировать часы" << std::endl;
            std::cout << "0 - Выйти из программы" << std::endl;

            int nCommand;
            std::cin >> nCommand;
            std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
            if (std::cin.fail()) {
                std::cin.clear();
                nCommand = -1;
            }

            switch (nCommand) {
            case 0:
                return 0;

            case 1:
                ShowSchedule(oController);
                break;

            case 2:
                SetTimezone(oController);
                break;

            case 3:
                ShowClock(oController);
                break;

            case 4:
                SyncClock(oController);
                break;

            default:
                std::cout << "Неправильный ввод" << std::endl;
                break;
            }
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
