#include "CScheduleDialog.h"

#include <gtkmm/version.h>

#include <codecvt>  // для std::wstring_convert
#include <iomanip>  // для std::put_time
#include <iostream>
#include <sstream>  // для std::stringstream

#include "utils.h"

#define HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY GTKMM_CHECK_VERSION(4, 9, 1)

const char32_t s_szDowTitles[] = U"ПВСЧПСВ";

CScheduleDialog::CScheduleDialog() :
    m_fClockValid(false),
    m_nCtrTimeSpan(0),
    m_tCtrTime(0),
    m_tPcTime(0),
    m_fClockFilled(false),
    m_nClockUpdateCounter(0),
    m_VBox(Gtk::Orientation::VERTICAL),
    m_ClockFrame("Часы контроллера"),
    m_CtrTimeLabel("Время контроллера"),
    m_PcTimeLabel("Время ПК"),
    m_SyncButton("Синхронизировать"),
    m_AutoSyncCheckButton("Авто синхронизация (>=5 с)"),
    m_TzFrame("Временные зоны"),
    m_InTzFrame("Вход"),
    m_aInTzRows{CMyListRow("1 _ _ _ _ _ _  __:__-__:__"), CMyListRow("2 _ _ _ _ _ _  __:__-__:__"),
                CMyListRow("3 _ _ _ _ _ _  __:__-__:__"), CMyListRow("4 _ _ _ _ _ _  __:__-__:__"),
                CMyListRow("5 _ _ _ _ _ _  __:__-__:__"), CMyListRow("6 _ _ _ _ _ _  __:__-__:__"),
                CMyListRow("7 _ _ _ _ _ _  __:__-__:__")},
    m_OutTzFrame("Выход"),
    m_aOutTzRows{CMyListRow("1 _ _ _ _ _ _  __:__-__:__"), CMyListRow("2 _ _ _ _ _ _  __:__-__:__"),
                 CMyListRow("3 _ _ _ _ _ _  __:__-__:__"), CMyListRow("4 _ _ _ _ _ _  __:__-__:__"),
                 CMyListRow("5 _ _ _ _ _ _  __:__-__:__"), CMyListRow("6 _ _ _ _ _ _  __:__-__:__"),
                 CMyListRow("7 _ _ _ _ _ _  __:__-__:__")},
    m_ModeTzFrame("Переключение режима контроллера"),
    m_aModeTzRows{CMyListRow("1 _ _ _ _ _ _  __:__-__:__"),
                  CMyListRow("2 _ _ _ _ _ _  __:__-__:__")} {
    set_destroy_with_parent(true);
    set_title("Расписание");
    set_default_size(600, 480);

    m_VBox.set_margin(5);
    set_child(m_VBox);
    m_VBox.append(m_ClockFrame);
    m_VBox.append(m_TzFrame);
    // m_VBox.append(m_ModeTzFrame);

    m_ClockFrame.set_child(m_ClockGrid);
    m_ClockGrid.set_margin(5);
    m_ClockGrid.set_row_spacing(10);
    m_ClockGrid.set_column_spacing(10);
    m_ClockGrid.attach(m_CtrTimeLabel, 0, 0);
    m_ClockGrid.attach(m_CtrTimeEntry, 1, 0);
    m_ClockGrid.attach(m_SyncButton, 2, 0);
    m_ClockGrid.attach_next_to(m_PcTimeLabel, Gtk::PositionType::BOTTOM);
    m_ClockGrid.attach(m_PcTimeEntry, 1, 1);
    m_ClockGrid.attach(m_AutoSyncCheckButton, 2, 1);
    m_CtrTimeLabel.set_halign(Gtk::Align::END);
    m_PcTimeLabel.set_halign(Gtk::Align::END);
    m_CtrTimeEntry.set_editable(false);
    m_CtrTimeEntry.set_alignment(Gtk::Align::CENTER);
    m_CtrTimeEntry.set_hexpand();
    m_CtrTimeEntry.set_size_request(170);
    m_PcTimeEntry.set_editable(false);
    // m_PcTimeEntry.set_halign(Gtk::Align::CENTER);
    m_PcTimeEntry.set_alignment(Gtk::Align::CENTER);
    m_PcTimeEntry.set_hexpand();
    m_PcTimeEntry.set_size_request(170);

    m_TzFrame.set_child(m_TzGrid);
    m_TzGrid.set_margin(5);
    m_TzGrid.set_row_spacing(10);
    m_TzGrid.set_column_spacing(10);
    m_TzGrid.attach(m_InTzFrame, 0, 0);
    m_TzGrid.attach(m_OutTzFrame, 1, 0);
    m_TzGrid.attach_next_to(m_ModeTzFrame, Gtk::PositionType::BOTTOM);

    m_InTzFrame.set_child(m_InTzListBox);
    m_InTzListBox.set_margin(5);
    m_InTzListBox.set_activate_on_single_click(false);
    for (size_t i = 0; i < std::size(m_aInTzRows); i++)
        m_InTzListBox.append(m_aInTzRows[i]);
    m_InTzListBox.signal_row_activated().connect(
        sigc::mem_fun(*this, &CScheduleDialog::on_in_tz_row_activated));

    m_OutTzFrame.set_child(m_OutTzListBox);
    m_OutTzListBox.set_margin(5);
    m_OutTzListBox.set_activate_on_single_click(false);
    for (size_t i = 0; i < std::size(m_aOutTzRows); i++)
        m_OutTzListBox.append(m_aOutTzRows[i]);
    m_OutTzListBox.signal_row_activated().connect(
        sigc::mem_fun(*this, &CScheduleDialog::on_out_tz_row_activated));

    m_ModeTzFrame.set_child(m_ModeTzListBox);
    m_ModeTzListBox.set_margin(5);
    m_ModeTzListBox.set_activate_on_single_click(false);
    for (size_t i = 0; i < std::size(m_aModeTzRows); i++)
        m_ModeTzListBox.append(m_aModeTzRows[i]);
    m_ModeTzListBox.signal_row_activated().connect(
        sigc::mem_fun(*this, &CScheduleDialog::on_mode_tz_row_activated));

    // Load extra CSS file.
    m_refCssProvider = Gtk::CssProvider::create();
#if HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY
    Gtk::StyleProvider::add_provider_for_display(get_display(), m_refCssProvider,
                                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#else
    Gtk::StyleContext::add_provider_for_display(get_display(), m_refCssProvider,
                                                GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#endif
    std::string sCSS =
        ".clock {"
        "  font: 100% Monospace, Courier New, Courier; "
        "}"
        ".filled {"
        "  color: red;"
        "  font-weight: bold;"
        "}"
        ".tz {"
        "  font: 100% Monospace, Courier New, Courier; "
        "}";

    m_refCssProvider->load_from_data(sCSS);
    m_PcTimeEntry.add_css_class("clock");
    m_CtrTimeEntry.add_css_class("clock");
    m_InTzListBox.add_css_class("tz");
    m_OutTzListBox.add_css_class("tz");
    m_ModeTzListBox.add_css_class("tz");

    m_SyncButton.signal_clicked().connect(sigc::mem_fun(*this, &CScheduleDialog::on_button_sync));
    m_AutoSyncCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CScheduleDialog::on_button_auto_sync));
    m_connection_timeout =
        Glib::signal_timeout().connect(sigc::mem_fun(*this, &CScheduleDialog::on_timeout), 333);
}

CScheduleDialog::~CScheduleDialog() {
    m_oController = nullptr;
}

void CScheduleDialog::Init(const ilg::CController& oController) {
    m_oController = oController.Clone();

    ilg_controller_info rInfo;
    m_oController.GetControllerInfo(rInfo);

    auto fDualZones = (rInfo.nCtrFlags & (ILG_CTR_F_TWOBANKS | ILG_CTR_F_DUALZONE)) != 0;
    m_OutTzFrame.set_visible(fDualZones);
    m_InTzFrame.set_label(fDualZones ? "Вход" : "Вход и выход");

    ilg_controller_options rOptions;
    m_oController.GetOptions(rOptions);
    m_AutoSyncCheckButton.set_active(static_cast<bool>(rOptions.fAutoSyncClock));

    CheckClock();
    m_oController.ReadTimeZones(0, 0, m_aInTzs, std::size(m_aInTzs));
    UpdateInTzView();
    if (m_OutTzFrame.get_visible()) {
        m_oController.ReadTimeZones(1, 0, m_aOutTzs, std::size(m_aOutTzs));
        UpdateOutTzView();
    }
    m_ModeTzFrame.set_visible((rInfo.nCtrFlags & ILG_CTR_F_MODES) != 0);
    if (m_ModeTzFrame.get_visible()) {
        m_oController.ReadModeTimeZones(0, m_aModeTzs, std::size(m_aModeTzs));
        UpdateModeTzView();
    }
}

void CScheduleDialog::on_button_sync() {
    m_oController.SyncClock();
    m_nCtrTimeSpan = 0;
    m_fClockValid = true;
    UpdateClock();
}

void CScheduleDialog::on_button_auto_sync() {
    auto fEnable = m_AutoSyncCheckButton.get_active();
    ilg_controller_options rOptions;
    m_oController.GetOptions(rOptions);
    rOptions.fAutoSyncClock = static_cast<ilg_bool>(fEnable);
    m_oController.SetOptions(rOptions);
}

bool CScheduleDialog::on_timeout() {
    if (++m_nClockUpdateCounter == 15) {
        m_nClockUpdateCounter = 0;
        CheckClock();
    }
    else
        UpdateClock();
    // As this is a timeout function, return true so that it continues to get called
    return true;
}

void CScheduleDialog::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void CScheduleDialog::on_in_tz_row_activated(Gtk::ListBoxRow* pRow) {
    DoChangeTz(false, pRow->get_index());
}

void CScheduleDialog::on_out_tz_row_activated(Gtk::ListBoxRow* pRow) {
    DoChangeTz(true, pRow->get_index());
}

void CScheduleDialog::on_mode_tz_row_activated(Gtk::ListBoxRow* pRow) {
    DoChangeModeTz(pRow->get_index());
}

void CScheduleDialog::on_tz_dialog_hide() {
    if (m_refTimeZoneDialog->m_fAccept) {
        try {
            if (m_refTimeZoneDialog->m_fModeTz) {
                m_oController.WriteModeTimeZones(m_refTimeZoneDialog->m_nIdx,
                                                 &m_refTimeZoneDialog->m_rModeTz, 1);
                m_aModeTzs[m_refTimeZoneDialog->m_nIdx] = m_refTimeZoneDialog->m_rModeTz;
                UpdateModeTzRow(m_refTimeZoneDialog->m_nIdx);
            }
            else if (m_refTimeZoneDialog->m_fOut) {
                m_oController.WriteTimeZones(1, m_refTimeZoneDialog->m_nIdx,
                                             &m_refTimeZoneDialog->m_rTz, 1);
                m_aOutTzs[m_refTimeZoneDialog->m_nIdx] = m_refTimeZoneDialog->m_rTz;
                UpdateOutTzRow(m_refTimeZoneDialog->m_nIdx);
            }
            else {
                m_oController.WriteTimeZones(0, m_refTimeZoneDialog->m_nIdx,
                                             &m_refTimeZoneDialog->m_rTz, 1);
                m_aInTzs[m_refTimeZoneDialog->m_nIdx] = m_refTimeZoneDialog->m_rTz;
                UpdateInTzRow(m_refTimeZoneDialog->m_nIdx);
            }
        }
        catch (const std::exception& e) {
            ShowMessage(e.what(), Gtk::MessageType::ERROR);
            std::cerr << e.what() << '\n';
        }
    }
    m_refTimeZoneDialog = nullptr;
}

void CScheduleDialog::UpdateClock(bool fForce) {
    auto tPcTime = time(nullptr);
    auto tCtrTime = (m_fClockValid ? (tPcTime + m_nCtrTimeSpan) : 0);

    if ((m_tCtrTime != tCtrTime) || fForce) {
        m_tCtrTime = tCtrTime;
        if (tCtrTime != 0)
            m_CtrTimeEntry.set_text(TimeToStr(tCtrTime));
        else
            m_CtrTimeEntry.set_text("???");
    }
    if ((m_tPcTime != tPcTime) || fForce) {
        m_tPcTime = tPcTime;
        m_PcTimeEntry.set_text(TimeToStr(tPcTime));
    }
    // Выделяем жирным если время рассинхронизировалось
    if ((std::abs(m_nCtrTimeSpan) > 5) != m_fClockFilled) {
        m_fClockFilled = !m_fClockFilled;
        if (m_fClockFilled)
            m_CtrTimeEntry.add_css_class("filled");
        else
            m_CtrTimeEntry.remove_css_class("filled");
    }
}

void CScheduleDialog::CheckClock() {
    try {
        ilg_rtc_params rRtc;
        m_oController.ReadRtcParams(rRtc);
        m_fClockValid = (rRtc.nDiffSec != std::numeric_limits<int64_t>::max());
        if (m_fClockValid)
            m_nCtrTimeSpan = rRtc.nDiffSec;
        UpdateClock(true);
    }
    catch (const std::exception& e) {
        if (nullptr == m_refDialog)
            ShowMessage(e.what(), Gtk::MessageType::ERROR);
        std::cerr << e.what() << '\n';
    }
}

std::string CScheduleDialog::TimeToStr(time_t t) {
    std::stringstream ss;
    auto tmb = std::localtime(&t);
    ss << std::put_time(tmb, "%d.%m.%Y %H:%M:%S");
    return ss.str();
}

void CScheduleDialog::ShowMessage(const std::string& sMessage, Gtk::MessageType nType) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    m_refDialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CScheduleDialog::on_dialog_response)));
}

void CScheduleDialog::UpdateInTzView() {
    for (size_t i = 0; i < std::size(m_aInTzs); i++)
        UpdateInTzRow(i);
}

void CScheduleDialog::UpdateInTzRow(size_t nIdx) {
    auto pTz = &m_aInTzs[nIdx];
    size_t nPos = 0;
    char32_t szDows[15] = {0};
    for (size_t k = 0; k < 7; k++) {
        szDows[nPos++] = (GET_BIT(pTz->nDaysOfWeek, k) ? s_szDowTitles[k] : '-');
        szDows[nPos++] = ' ';
    }
    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
    std::stringstream ss;
    ss << (nIdx + 1) << "  " << cv.to_bytes(szDows) << "  " << std::setfill('0') << std::setw(2)
       << (uint)pTz->nBeginHour << ':' << std::setw(2) << (uint)pTz->nBeginMinute << '-'
       << std::setw(2) << (uint)pTz->nEndHour << ':' << std::setw(2) << (uint)pTz->nEndMinute;
    m_aInTzRows[nIdx].set_text(ss.str());
}

void CScheduleDialog::UpdateOutTzView() {
    for (size_t i = 0; i < std::size(m_aOutTzs); i++)
        UpdateOutTzRow(i);
}

void CScheduleDialog::UpdateOutTzRow(size_t nIdx) {
    auto pTz = &m_aOutTzs[nIdx];
    size_t nPos = 0;
    char32_t szDows[15] = {0};
    for (size_t k = 0; k < 7; k++) {
        szDows[nPos++] = (GET_BIT(pTz->nDaysOfWeek, k) ? s_szDowTitles[k] : '-');
        szDows[nPos++] = ' ';
    }
    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
    std::stringstream ss;
    ss << (nIdx + 1) << "  " << cv.to_bytes(szDows) << "  " << std::setfill('0') << std::setw(2)
       << (uint)pTz->nBeginHour << ':' << std::setw(2) << (uint)pTz->nBeginMinute << '-'
       << std::setw(2) << (uint)pTz->nEndHour << ':' << std::setw(2) << (uint)pTz->nEndMinute;
    m_aOutTzRows[nIdx].set_text(ss.str());
}

void CScheduleDialog::UpdateModeTzView() {
    for (size_t i = 0; i < std::size(m_aModeTzs); i++)
        UpdateModeTzRow(i);
}

void CScheduleDialog::UpdateModeTzRow(size_t nIdx) {
    auto pTz = &m_aModeTzs[nIdx];
    size_t nPos = 0;
    char32_t szDows[15] = {0};
    for (size_t k = 0; k < 7; k++) {
        szDows[nPos++] = (GET_BIT(pTz->nDaysOfWeek, k) ? s_szDowTitles[k] : '-');
        szDows[nPos++] = ' ';
    }
    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
    std::stringstream ss;
    ss << (nIdx + 1) << "  " << cv.to_bytes(szDows) << "  " << std::setfill('0') << std::setw(2)
       << (uint)pTz->nBeginHour << ':' << std::setw(2) << (uint)pTz->nBeginMinute << '-'
       << std::setw(2) << (uint)pTz->nEndHour << ':' << std::setw(2) << (uint)pTz->nEndMinute
       << "  " << kControllerModeNames[pTz->nMode];
    m_aModeTzRows[nIdx].set_text(ss.str());
}

void CScheduleDialog::DoChangeTz(bool fOut, size_t nIdx) {
    if (nullptr == m_refTimeZoneDialog)
        m_refTimeZoneDialog.reset(new CTimeZoneDialog());
    m_refTimeZoneDialog->set_transient_for(*this);
    m_refTimeZoneDialog->set_modal();
    m_refTimeZoneDialog->set_hide_on_close();
    m_refTimeZoneDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CScheduleDialog::on_tz_dialog_hide));

    m_refTimeZoneDialog->m_rTz = fOut ? m_aOutTzs[nIdx] : m_aInTzs[nIdx];
    m_refTimeZoneDialog->SetTzIdx(false, fOut, nIdx);

    m_refTimeZoneDialog->set_visible(true);
}

void CScheduleDialog::DoChangeModeTz(size_t nIdx) {
    if (nullptr == m_refTimeZoneDialog)
        m_refTimeZoneDialog.reset(new CTimeZoneDialog());
    m_refTimeZoneDialog->set_transient_for(*this);
    m_refTimeZoneDialog->set_modal();
    m_refTimeZoneDialog->set_hide_on_close();
    m_refTimeZoneDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CScheduleDialog::on_tz_dialog_hide));

    m_refTimeZoneDialog->m_rModeTz = m_aModeTzs[nIdx];
    m_refTimeZoneDialog->SetTzIdx(true, false, nIdx);

    m_refTimeZoneDialog->set_visible(true);
}
