/**
 * @file ilreaders.h
 * @brief Заголовочный файл API SDK Readers.
 *
 * Данный файл содержит в себе определение API SDK Readers.
 * SDK Readers - библиотека для связи с настольными и сетевыми
 * считывателями Iron Logic.
 * Библиотека предназначена для ОС Linux.
 * [Сайт SDK](https://ironlogic.ru/il_new.nsf/htm/ru_sdk-readers)
 */
/**
 * @defgroup init Инициализация, версия и коды ошибок.
 * @brief Функции и типы, связанные с инициализацией и обработкой ошибок.
 *
 * Ссылки на инфо о инициализации и завершении библиотеки, проверке версии и обработке ошибок.
 */
/**
 * @defgroup search Поиск считывателей.
 * @brief Функции и типы, связанные с поиском считывателей.
 *
 * Ссылки на инфо о создании дескриптора поиска, поиске считывателей, настройке поиска.
 */
/**
 * @defgroup reader Связь со считывателем.
 * @brief Функции и типы, связанные с связью со считывателем.
 *
 * Ссылки на инфо о создании дескриптора считывателя, настройке параметров связи со считывателем,
 * подключение к считывателю, сканирование карт, чтение/запись данных карты.
 */
#ifndef ILREADERS_H_INCLUDED
#define ILREADERS_H_INCLUDED
#include <stdint.h>  // для uint8_t
#include <string.h>  // для memcpy, memset и memcmp

/// Переключатель между статическим и динамическим связыванием.
// #define ILREADERS_LINKONREQUEST

#ifndef ILREADERS_LINKONREQUEST
#ifdef ILR_STATIC_LIB
#define ILR_API
#elif defined(_WIN32) || defined(__CYGWIN__)
#ifdef ILREADERS_EXPORTS
#define ILR_API __declspec(dllexport)
#else
#define ILR_API __declspec(dllimport)
#endif
#else  // non Windows
#define ILR_API
#endif  // ILR_STATIC_LIB
#endif  // !ILREADERS_LINKONREQUEST

/// Макрос, определяющий соглашение о вызове функций.
#if defined(_WIN32) || defined(__CYGWIN__)
#define ILR_CALL WINAPI
#else
#define ILR_CALL
#endif /* _WIN32 || __CYGWIN__ */

/** @name Макросы версии SDK Readers.
 * @{ */
/**
 * @brief Основной номер версии библиотеки SDK Readers.
 *
 * Увеличивается, когда API становится не совместимым с предыдущими версиями.
 * @ingroup init
 */
#define ILR_VERSION_MAJOR 1
/**
 * @brief Минорный номер версии библиотеки SDK Readers.
 *
 * Увеличивается, когда в API добавляются новые возможности,
 * но при этом сохраняется обратная совместимость.
 * @ingroup init
 */
#define ILR_VERSION_MINOR 0
/**
 * @brief Номер патча библиотеки SDK Readers.
 *
 * Увеличивается, когда выпускается исправление ошибки,
 * не содержащее никаких изменений в API.
 * @ingroup init
 */
#define ILR_VERSION_PATCH 2
/**
 * @brief Полный номер версии библиотеки SDK Readers.
 * @ingroup init
 */
#define ILR_VERSION \
    ((ILR_VERSION_MAJOR << 24) | (ILR_VERSION_MINOR << 16) | (ILR_VERSION_PATCH << 8) | 0xC0)

/** Извлекает основной номер из полной версии. */
#define ILR_VERSION_GET_MAJOR(v) ((v) >> 24)
/** Извлекает минорный номер из полной версии. */
#define ILR_VERSION_GET_MINOR(v) (((v) >> 16) & 0xff)
/** Извлекает номер патча из полной версии. */
#define ILR_VERSION_GET_PATCH(v) (((v) >> 8) & 0xff)
/** @} */

/**
 * @brief Константа, указывающая на необходимость выделить память автоматически.
 *
 * Используется в функциях, которые возвращают строки и массивы.
 * Для освобождения памяти нужно использовать функцию @ref ilr_free_memory.
 */
#define ILR_AUTOALLOCATE 0xffffffff

/** Тип дескриптора (ссылка на объект библиотеки). */
typedef void* ilr_handle;

/** Флаг. */
typedef enum : int {
    ILR_FALSE,  ///< Ложь. =0
    ILR_TRUE  ///< Истина. Может быть любое число отличное от 0.
} ilr_bool;

/**
 * @brief Коды ошибок.
 * @ingroup init
 * @{
 */
typedef enum {
    /// Операция успешно выполнена.
    ILR_OK = 0,
    /// Неизвестная ошибка.
    ILR_E_FAIL,
    /// Операция выполняется асинхронно.
    ILR_E_PENDING,
    /// Индекс вне диапазона.
    ILR_E_BOUNDS,
    /// Команда не поддерживается считывателем.
    ILR_E_NOTIMPL,
    /// Неправильный указатель.
    ILR_E_POINTER,
    /// Операция прервана.
    ILR_E_ABORT,
    /// Неправильный дескриптор.
    ILR_E_HANDLE,
    /// Недостаточно памяти.
    ILR_E_OUTOFMEMORY,
    /// Неправильные параметры.
    ILR_E_INVALIDARG,
    /// Неправильная версия Sdk Readers передана в функцию @ref ilr_init.
    ILR_E_WRONG_SDK_VERSION,
    /// Библиотека не инициализирована с помощью @ref ilr_init.
    ILR_E_NOT_INITIALIZED,
    /// Недостаточно ресурсов для завершения операции.
    ILR_E_OUT_OF_RESOURCES,
    /// Размер буфера слишком мал
    ILR_E_BUFFER_TOO_SMALL,
    /// Блокирующий вызов функции из обработчика не разрешен.
    ILR_E_BLOCKING_CALL_NOT_ALLOWED,
    /// Порт не существует.
    ILR_E_PORT_NOT_EXIST,
    /// Нет доступа к порту.
    ILR_E_PORT_ACCESS_DENIED,
    /// Неизвестная ошибка открытия порта.
    ILR_E_PORT_OPEN_FAILED,
    /// Ошибка подключения.
    ILR_E_CONNECTION_ERROR,
    /// Связь с устройством была потеряна.
    ILR_E_CONNECTION_DOWN,
    /// Тайм-аут запроса к считывателю.
    ILR_E_REQUEST_TIMEOUT,
    /// Не распознан ответ считывателя.
    ILR_E_BAD_RESPONSE,
    /// Ошибка функции Smart Cards.
    ILR_E_SCARD_ERROR,
    /// Ошибка считывателя.
    ILR_E_READER_ERROR,
    /// Нет карты.
    ILR_E_NO_CARD,
    /// Карта отказала от выполнения команды.
    ILR_E_CARD_NACK,
    /// Ошибка авторизации карты.
    ILR_E_CARD_AUTH,
    /// Неправильный адрес Mifare.
    ILR_E_MIFARE_ADDRESS,
    /// Ошибка блока-значения Mifare.
    ILR_E_MIFARE_VALUE,
    /// Страница карты заблокирована.
    ILR_E_CARD_PAGE_LOCK,
    /// Не удалось записать на Temic.
    ILR_E_WRITE_T57,
    /// Размер списка ошибок.
    ILR_STATUS_SIZE
} ilr_status;

/**
 * @brief Проверяет код возврата SDK на успех.
 *
 * @param[in] status Код возврата.
 * @ingroup init
 */
#define ILR_SUCCEEDED(status) ((status) == ILR_OK)

/**
 * @brief Проверяет код возврата SDK на ошибку.
 *
 * @param[in] status Код возврата.
 * @ingroup init
 */
#define ILR_FAILED(status) ((status) != ILR_OK)

/*! @} */

/**
 * @brief Модель считывателя.
 *
 * @sa @ref readers "Список моделей считывателей"
 */
typedef enum {
    /** Неизвестная модель. */
    ILR_READER_MODEL_UNKNOWN,
    /** [Z-2 (мод. RD_ALL) / Z-2 USB](@ref z2_rdall). */
    ILR_READER_MODEL_Z2_RDALL,
    /** [Z-2 (мод. MF) / Z-2 USB MF](@ref z2_usb_mf). */
    ILR_READER_MODEL_Z2_MF,
    /** [Z-2 (мод. MF-I)](@ref z2_mfi). */
    ILR_READER_MODEL_Z2_MFI,
    /** [Z-2 (мод. E HTZ RF) / Z-2 EHR](@ref z2_ehr). */
    ILR_READER_MODEL_Z2_E_HTZ_RF,
    /** [Z-1 (мод. N Z) / Z-2 Base](@ref z2_base). */
    ILR_READER_MODEL_Z1_N_Z,
    /** [Z-2 (мод. E HT Hotel) / Z-2 RF-1996](@ref z2_rf1996). */
    ILR_READER_MODEL_Z2_E_HT_HOTEL,
    /** [Matrix-III (мод. RD_All)](@ref matrix3_rdall). */
    ILR_READER_MODEL_MATRIX3_RDALL,
    /** [Matrix-III (мод. MF K Net) / Matrix-III Net](@ref matrix3_net). */
    ILR_READER_MODEL_MATRIX3_NET,
    /** [СP-Z-2 (мод. MF-I)](@ref cp_z2_mfi). */
    ILR_READER_MODEL_CP_Z2_MFI,
    /** [Matrix-V (мод. E S RF) / Matrix-V](@ref matrix5). */
    ILR_READER_MODEL_MATRIX5_E_S_RF,
    /** [Matrix-VI (мод. NFC K Net)](@ref matrix6). */
    ILR_READER_MODEL_MATRIX6,
    /** [Z-2 (мод. MF CCID)](@ref z2_mf_ccid). */
    ILR_READER_MODEL_Z2_MF_CCID,
    /** Размер списка. */
    ILR_READER_MODEL_SIZE
} ilr_reader_model;

/** Тип порта считывателя. */
typedef enum {
    /** Не известно. */
    ILR_PORT_TYPE_UNKNOWN,
    /** Имя последовательного порта (например "/dev/ttyACM0"). */
    ILR_PORT_TYPE_COMPORT,
    /** CCID (Smart Cards) (например "IronLogic Z-2 CCID (01031) 00 00"). */
    ILR_PORT_TYPE_CCID,
    /** Адрес конвертера в режиме "Сервер" (например "10.0.0.2:1000"). */
    ILR_PORT_TYPE_SERVER,
    /** Адрес конвертера в режиме "Клиент" (например "40200@25000"). */
    ILR_PORT_TYPE_CLIENT,
    /** Адрес конвертера в режиме "Прокси" (например "40200:36D782FB@127.0.0.1:25001") */
    ILR_PORT_TYPE_PROXY,
    /** Размер списка. */
    ILR_PORT_TYPE_SIZE
} ilr_port_type;

/// Тип карты.
typedef enum {
    ILR_CARD_UNKNOWN,     ///< Не известно.
    ILR_CARD_EM_MARINE,   ///< Em-Marine.
    ILR_CARD_HID,         ///< HID.
    ILR_CARD_ICODE,       ///< iCode.
    ILR_CARD_COD433,      ///< Cod433.
    ILR_CARD_COD433_FIX,  ///< Cod433 Fix.
    ILR_CARD_CAME433,     ///< Радио-брелок CAME.
    ILR_CARD_DALLAS,      ///< Dallas.
    ILR_CARD_TEMIC,       ///< [Temic (T5557)](@ref temic).

    ILR_CARD_MF_ULTRALIGHT,    ///< [Mifare UltraLight](@ref mf_ultralight).
    ILR_CARD_MF_ULTRALIGHT_C,  ///< Mifare Ultralight C.
    ILR_CARD_MF_MINI,          ///< Mifare Mini.
    ILR_CARD_MF_CLASSIC_1K,    ///< [Mifare Classic 1K](@ref mf_classic_1k).
    ILR_CARD_MF_CLASSIC_2K,    ///< Mifare Classic 2K.
    ILR_CARD_MF_CLASSIC_4K,    ///< [Mifare Classic 4K](@ref mf_classic_4k).
    ILR_CARD_MF_PLUS,          ///< [Mifare Plus](@ref mf_plus).
    ILR_CARD_MF_PLUS_1K,       ///< Mifare Plus 1K.
    ILR_CARD_MF_PLUS_2K,       ///< Mifare Plus 2K.
    ILR_CARD_MF_PLUS_4K,       ///< Mifare Plus 4K.
    ILR_CARD_SMART_MX_MF1K,    ///< Smart MX with Mifare 1K.
    ILR_CARD_SMART_MX_MF4K,    ///< Smart MX with Mifare 4K.
    ILR_CARD_MF_DESFIRE,       ///< Mifare DESFire.
    ILR_CARD_MF_PROX,          ///< Mifare ProX.
    ILR_CARD_TYPE_SIZE         ///< Размер списка.
} ilr_card_type;

/**
 * @struct ilr_reader_info
 * @brief Информация о считывателе.
 *
 * Описывает основную информацию о считывателе.
 */
typedef struct {
    /** Тип порта. */
    ilr_port_type nPortType;
    /** Имя порта. */
    const char* pszPortName;
    /** Для IP конвертера: IP или имя сервера, к которому подключён конвертер. Если пустая строка,
     * то порт свободен. */
    const char* pszConnect;
    /** Модель считывателя. */
    ilr_reader_model nModel;
    /** Серийный номер считывателя. Если =-1, то не известно. */
    int nSn;
    /** Версия прошивки считывателя. Если =0, то не известно. Если старший байт =0, то номер версий
     * это одно число, иначе каждый байт представляет собой номер версии, начиная от старшего байта.
     */
    uint32_t nFwVersion;
    /** Дата и время сборки прошивки. Если =0, то не известно. Эквивалентно типу __time64_t в C++ -
     * количество секунд с 00:00:00 1 января 1970 года. */
    int64_t nFwBuildDate;
    /** Статус идентификации считывателя. */
    ilr_status nError;
} ilr_reader_info;

/// Уровень лога.
typedef enum {
    /**
     * @brief Лог выключен.
     * @details Лог сообщения не показываются.
     */
    ILR_LOG_LEVEL_NONE,

    /**
     * @brief Ошибки.
     * @details Показывает любые ошибки, которые являются фатальными для операции, но не для службы
     * или приложения.
     */
    ILR_LOG_LEVEL_ERROR,

    /**
     * @brief Предупреждения.
     * @details Показывает возможные проблемы, которые не являются ошибками.
     */
    ILR_LOG_LEVEL_WARNING,

    /**
     * @brief Уведомления.
     * @details Показывает полезную информацию, в основном успехи.
     */
    ILR_LOG_LEVEL_INFO,

    /**
     * @brief Отладочные сообщения.
     * @details Показывает шаги программы, получаемые и отправляемые данные.
     */
    ILR_LOG_LEVEL_DEBUG,
    /** Размер списка. */
    ILR_LOG_LEVEL_SIZE
} ilr_log_level;

/**
 * @brief Тип функции обратного вызова для получения сообщений лога отладки библиотеки.
 *
 * Функция имеет следующий вид:
 * @code
 * void logging_callback(ilr_log_level level, const char *pContext, const char *pMessage, void
 * *pUserData)
 * @endcode
 *
 * @param[in] nLevel    Уровень лога.
 * @param[in] pContext  Контекст.
 * @param[in] pMessage  Текст сообщения.
 * @param[in] pUserData Указатель на данные пользователя.
 *
 * @ingroup init
 */
typedef void(ILR_CALL* ilr_logging_callback)(ilr_log_level nLevel, const char* pContext,
                                             const char* pMessage, void* pUserData);

/**
 * @brief Настройки библиотеки.
 *
 * @ingroup init
 */
typedef struct {
    /**
     * Посылать два стоповых бита, иначе один (для COM-порта). По умолчанию ILR_FALSE.
     */
    ilr_bool fTwoStopBits;
    /**
     * Тайм-аут запроса в миллисекундах. По умолчанию 3000.
     */
    int nRequestTimeout;
    /**
     * Количество попыток запроса. По умолчанию 2.
     */
    int nRequestAttempts;
} ilr_options;

/**
 * @brief Тип функции обратного вызова для фильтрации портов при поиске считывателей.
 *
 * @param[in] nPortType    Тип порта.
 * @param[in] pszPortName  Имя порта.
 * @param[in] pUserData    Указатель на данные пользователя.
 *
 * @return ILR_TRUE, порт нужно исключить.
 * @ingroup search
 */
typedef ilr_bool(ILR_CALL* ilr_filter_port_callback)(ilr_port_type nPortType,
                                                     const char* pszPortName, void* pUserData);

/**
 * @brief Сообщение поиска считывателей.
 * @ingroup search
 */
typedef enum {
    /**
     * Завершилась асинхронная команда, созданная функцией с префиксом "ilr_search_begin_".
     * Параметр - дескриптор команды (дескриптор действителен до следующего сообщения этого типа или
     * до следующего вызова GetMessage), его не нужно закрывать функцией @ref ilr_close_handle.
     */
    ILR_SEARCH_MSG_COMMAND_FINISH,
    /**
     * Считыватель найден, параметр @ref ilr_reader_info*.
     */
    ILR_SEARCH_MSG_READER_FOUND,
    /**
     * Считыватель потерян, параметр @ref ilr_reader_info*.
     */
    ILR_SEARCH_MSG_READER_LOST,
    /**
     * Список считывателей изменён.
     * @remark Это сообщение посылается при добавлении/удалении считывателя в список найденных и при
     * изменении параметра считывателя @ref ilr_reader_info.
     */
    ILR_SEARCH_MSG_LIST_CHANGED
} ilr_search_msg;

/**
 * @brief Тип функции обратного вызова для получения уведомлений от дескриптора поиска.
 *
 * @param[in] nMsg      Тип сообщения.
 * @param[in] pMsgData  Указатель на данные сообщения. Тип данных зависит от типа сообщения.
 * @param[in] pUserData Указатель на данные пользователя.
 *
 * @warning Указатель @p pMsgData действителен до выхода из этой функции.
 *
 * @ingroup search
 */
typedef void(ILR_CALL* ilr_search_message_callback)(ilr_search_msg nMsg, const void* pMsgData,
                                                    void* pUserData);

/**
 * @brief Флаги типов считывателей.
 * @ingroup search
 */
typedef enum : uint32_t {
    /** USB считыватели Iron Logic. */
    ILR_READER_ILUSB = (1 << 0),
    /**
     * USB считыватели сторонних производителей.
     * Успешная работа с такими считывателями не гарантируется.
     */
    ILR_READER_TPUSB = (1 << 1),
    /** IP конвертеры в режиме "Сервер" (поиск по UDP). */
    ILR_READER_SERVER = (1 << 2),
    /** IP конвертеры в режиме "Клиент" (прослушка TCP). */
    ILR_READER_CLIENT = (1 << 3),
    /**
     * Считыватели SmartCards.
     * Из считывателей Iron Logic поддерживается только считыватель Z-2 MF CCID.
     */
    ILR_READER_CCID = (1 << 4)
} ilr_reader_types;

/**
 * @brief Настройки поиска считывателей.
 * @ingroup search
 */
typedef struct {
    /**
     * Типы считывателей, которые нужно искать.
     * По умолчанию (ILR_READER_ILUSB | ILR_READER_SERVER | ILR_READER_CLIENT).
     */
    uint32_t nReaderTypes;
    /** Период между попытками открыть порт в миллисекундах. По умолчанию 1000. */
    int nOpenPortPeriod;
    /** Период между попытками опросить порт в миллисекундах. По умолчанию 60000. */
    int nPollPeriod;
    /** Период опроса IP конвертеров по UDP в миллисекундах. По умолчанию 5000. */
    int nUdpScanPeriod;
    /** Тайм-аут запроса по UDP в миллисекундах. По умолчанию 1500. */
    int nUdpRequestTimeout;
    /** Количество попыток запроса по UDP. По умолчанию 2. */
    int nUdpRequestAttempts;
    /** Период между попытками открыть TCP порт для прослушки конвертеров в режиме "Клиент" в
     * миллисекундах. По умолчанию 3000. */
    int nOpenListenerPeriod;
    /** Тайм-аут ожидания подключения конвертера в режиме "Клиент" после открытия порта прослушки.
     * Z-397 Web делает попытки примерно 1 раз в секунду. По умолчанию 1800. */
    int nAcceptTimeout;
} ilr_search_options;

/**
 * @brief Сообщение считывателя.
 * @ingroup reader
 */
typedef enum {
    /**
     * Завершилась асинхронная команда, созданная функцией с префиксом "ilr_reader_begin_".
     * Параметр - дескриптор команды (дескриптор действителен до следующего сообщения этого типа или
     * до следующего вызова GetMessage), его не нужно закрывать функцией @ref ilr_close_handle.
     */
    ILR_READER_MSG_COMMAND_FINISH,
    /**
     * Изменилось состояние подключения к считывателю.
     * Узнать текущее состояние можно функцией @ref ilr_reader_get_connection_status.
     */
    ILR_READER_MSG_CONNECTION_CHANGED,
    /**
     * Карта найдена, параметр @ref ilr_card_info*.
     */
    ILR_READER_MSG_CARD_FOUND,
    /**
     * Карта потеряна, параметр @ref ilr_card_info*.
     */
    ILR_READER_MSG_CARD_LOST
} ilr_reader_msg;

/**
 * @brief Тип функции обратного вызова для получения уведомлений от дескриптора считывателя.
 *
 * @param[in] nMsg      Тип сообщения.
 * @param[in] pMsgData  Указатель на данные сообщения. Тип данных зависит от типа сообщения.
 * @param[in] pUserData Указатель на данные пользователя.
 *
 * @warning Указатель @p pMsgData действителен до выхода из этой функции.
 *
 * @ingroup reader
 */
typedef void(ILR_CALL* ilr_reader_message_callback)(ilr_reader_msg nMsg, const void* pMsgData,
                                                    void* pUserData);

/**
 * @brief Состояние подключения к считывателю.
 * @ingroup reader
 */
typedef enum {
    ILR_CONNECTION_DISCONNECTED,  ////< Отключён.
    ILR_CONNECTION_CONNECTED,     ///< Подключён.
    ILR_CONNECTION_CONNECTING     ///< Подключение.
} ilr_connection_status;

/**
 * @brief ID карты
 * @ingroup reader
 */
typedef struct __attribute__((packed)) {
    uint8_t nLength;  ///< Длина номера в байтах.

    union {
        uint8_t aBytes[15];  ///< Байты номера.

        struct __attribute__((packed)) {
            uint16_t nNumber;
            uint8_t nSeries;
            uint16_t nFacility;
        } em_marine;
    };
} ilr_card_uid;

/**
 * @brief Уровень безопасности Mifare Plus.
 * @ingroup reader
 */
typedef enum : int8_t {
    ILR_MF_PLUS_SL_UNKNOWN = -1,  ///< Не известно.
    ILR_MF_PLUS_SL_SL0,           ///< Уровень 0 (не инициализирована).
    ILR_MF_PLUS_SL_SL1,           ///< Уровень 1 (эмуляция Mifare Classic).
    ILR_MF_PLUS_SL_SL2,           ///< Уровень 2.
    ILR_MF_PLUS_SL_SL3            ///< Уровень 3.
} ilr_mf_plus_sl;

/**
 * @brief Тип Mifare Plus.
 * @ingroup reader
 */
typedef enum : uint8_t {
    ILR_MF_PLUS_UNKNOWN,   ///< Не известно.
    ILR_MF_PLUS_S,         ///< Mifare Plus S.
    ILR_MF_PLUS_X,         ///< Mifare Plus X.
    ILR_MF_PLUS_SE,        ///< MIFARE Plus SE.
    ILR_MF_PLUS_EV1,       ///< MIFARE Plus EV1.
    ILR_MF_PLUS_EV2,       ///< MIFARE Plus EV2.
    ILR_MF_PLUS_TYPE_SIZE  ///< Размер списка.
} ilr_mf_plus_type;

/**
 * @brief Информация о карте.
 * @ingroup reader
 */
typedef struct {
    ilr_card_type nType;       ///< Тип карты.
    ilr_card_uid rUID;         ///< Номер карты.
    uint32_t nMemSize;         ///< Размер памяти карты (в байтах).
    ilr_mf_plus_sl nSL;        ///< Уровень безопасности Mifare Plus.
    ilr_mf_plus_type nMpType;  ///< Тип Mifare Plus.
    ilr_card_type nType2;      ///< Тип карты.
    ilr_card_uid rUID2;        ///< Номер карты.
} ilr_card_info;

/**
 * @brief Ключ аутентификации Mifare Classic.
 * @ingroup reader
 */
typedef int64_t ilr_mf_classic_key;

/**
 * @brief Ключ аутентификации Mifare Plus.
 * @ingroup reader
 */
typedef struct {
    union {
        uint8_t a[16];  ///< Байты ключа.

        struct {
            uint64_t lo;  ///< Младшая часть ключа.
            uint64_t hi;  ///< Старшая часть ключа.
        } ll;
    };
} ilr_mf_plus_key;

/**
 * @brief Данные блока Mifare Classic/Plus.
 * @ingroup reader
 */
typedef struct {
    uint8_t a[16] = {0};  ///< Байты блока.
} ilr_mf_block_data;

/**
 * @brief Флаги типов перезаписываемых карт.
 * @remark Для авто приостановки сканирования при обнаружении карты.
 * @ingroup reader
 */
typedef enum : uint32_t {
    ILR_RWCT_F_TEMIC = (1 << 0),          ///< Temic.
    ILR_RWCT_F_MF_ULTRALIGHT = (1 << 1),  ///< Mifare Ultralight.
    ILR_RWCT_F_MF_CLASSIC = (1 << 2),     ///< Mifare Classic или Plus SL1.
    ILR_RWCT_F_MF_PLUS_SL0 = (1 << 3),    ///< Mifare Plus SL0.
    ILR_RWCT_F_MF_PLUS_SL3 = (1 << 4),    ///< Mifare Plus SL3.
    ILR_RWCT_F_MF_PLUS = (1 << 5)         ///< Другие Mifare Plus
} ilr_rwcard_type_flags;

/**
 * Возвращает параметр сканирования Temic, объединяя тип @p type и скорость @p speed.
 * @ingroup reader
 * @{ */
#define MAKE_TEMIC_SCAN_PARAM(type, speed) (int)(((uint)((speed)&0xff) << 8) | ((type)&0xff))
/**
 * Возвращает тип, извлекая его из параметра сканирования Temic @p param.
 */
#define TYPE_FROM_TEMIC_SCAN_PARAM(param) (uint8_t)((param)&0xff)
/**
 * Возвращает скорость, извлекая её из параметра сканирования Temic @p param.
 */
#define SPEED_FROM_TEMIC_SCAN_PARAM(param) (uint8_t)(((uint)(param) >> 8) & 0xff)

/** @} */

/**
 * @brief Настройки считывателя.
 * @ingroup reader
 */
typedef struct {
    /** Модель считывателя для подключения. По умолчанию ILR_READER_MODEL_UNKNOWN. */
    ilr_reader_model nConnectModel;

    /** Типы карт, при обнаружении которых сканирование приостанавливается. По умолчанию 0. */
    uint32_t nHoldCardTypes;

    /** Период авто подключения к считывателю в миллисекундах. По умолчанию -1.
     * @remark Это общий параметр для всех считывателей.
     */
    int nReconnectPeriod;
} ilr_reader_options;

#ifdef __cplusplus
extern "C" {
#endif

#ifndef ILREADERS_LINKONREQUEST

/**
 * @brief Возвращает номер версии библиотеки.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilr_init.
 * @ingroup init
 */
ILR_API uint32_t ILR_CALL ilr_get_version();

/**
 * @brief Возвращает описание ошибки по её коду.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilr_init.
 *
 * @param[in] nErrorCode Код ошибки.
 *
 * @ingroup init
 */
ILR_API const char* ILR_CALL ilr_get_error_text(ilr_status nErrorCode);

/**
 * @brief Устанавливает уровень лога отладки.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilr_init.
 *
 * @param[in] nLevel Уровень лога отладка.
 *
 * @return Всегда код успеха `ILR_OK`.
 *
 * @ingroup init
 */
ILR_API ilr_status ILR_CALL ilr_set_log_level(ilr_log_level nLevel);

/**
 * @brief Устанавливает функцию обратного вызова для лога отладки.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilr_init.
 *
 * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать для передачи
 * сообщений лога отладки.
 * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в try catch.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе вернёт ошибку `ILR_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return Всегда код успеха `ILR_OK`.
 *
 * @ingroup init
 */
ILR_API ilr_status ILR_CALL ilr_set_log_callback(ilr_logging_callback pCallback,
                                                 void* pUserData = nullptr);

/**
 * @brief Освобождает память, которую выделила библиотека.
 *
 * Память могут выделять функции: @ref ilr_search_get_listen_ports, @ref ilr_reader_mf_rats.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilr_init.
 *
 * @param[in] p Указатель на память.
 *
 * @return Всегда код успеха `ILR_OK`.
 */
ILR_API ilr_status ILR_CALL ilr_free_memory(void* p);

/**
 * @brief Проверяет версию SDK.
 *
 * Возвращает `true` если версия библиотеки совместима с этим заголовочным файлом.
 * @ingroup init
 */
inline bool ilr_check_version() {
    auto v = ilr_get_version();
    return ((((v >> 24) & 0xff) == ILR_VERSION_MAJOR) &&
            ((int)((v >> 16) & 0xff) >= ILR_VERSION_MINOR));
}

/**
 * @brief Инициализирует библиотеку SDK Readers.
 *
 * Эта функция инициализирует библиотеку SDK Readers. Функция ilr_init должна быть вызвана первой
 * перед вызовом других функций, кроме @ref ilr_get_version, @ref ilr_get_error_text,
 * @ref ilr_set_log_level, @ref ilr_set_log_callback и @ref ilr_free_memory. Завершив использование
 * библиотеки приложение должно вызвать @ref ilr_cleanup, чтобы библиотека освободила внутренние
 * ресурсы. Приложение должно вызвать @ref ilr_cleanup при каждом успешном вызове ilr_init, т.к.
 * используется внутренний счётчик инициализаций.
 *
 * @remark Эта функция сама проверяет версию SDK, поэтому вызывать @ref ilr_check_version не нужно.
 *
 * @param[in] nVersionRequested Номер версии API SDK Readers, всегда должно быть равен
 * @ref ILR_VERSION.
 *
 * @return `ILR_OK` в случае успеха, иначе код ошибки.
 * @retval `ILR_E_WRONG_SDK_VERSION` запрошенная версия SDK не поддерживается.
 *
 * @sa @ref ilr_cleanup
 * @ingroup init
 */
ILR_API ilr_status ILR_CALL ilr_init(uint32_t nVersionRequested = ILR_VERSION);

/**
 * @brief Освобождает ресурсы библиотеки.
 *
 * Освобождает память и закрывает все дескрипторы библиотеки.
 *
 * @warning Эта функция не потокобезопасна. Её нельзя вызывать одновременно с функциями, которым
 * требуется инициализация библиотеки. Её нельзя вызывать из функции обратного вызова.
 *
 * @return Всегда код успеха `ILR_OK`.
 *
 * @sa @ref ilr_init
 * @ingroup init
 */
ILR_API ilr_status ILR_CALL ilr_cleanup();

/**
 * @brief Закрывает дескриптор библиотеки.
 *
 * @param[in] h Дескриптор библиотеки.
 */
ILR_API ilr_status ILR_CALL ilr_close_handle(ilr_handle h);

/**
 * @brief Клонирует дескриптор поиска считывателей или дескриптор считывателя.
 *
 * @param[in]  h           Исходный дескриптор.
 * @param[out] pNewHandle  Новый дескриптор.
 */
ILR_API ilr_status ILR_CALL ilr_clone_handle(ilr_handle h, ilr_handle* pNewHandle);

/**
 * @brief Устанавливает функцию обратного вызова для фильтрации портов при поиске считывателей.
 *
 * @param[in] pCallback Указатель на функцию, которую поток поиска будет вызывать при нахождении
 * порта.
 * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в try catch.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе вернёт ошибку `ILR_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_set_filter_port_callback(ilr_filter_port_callback pCallback,
                                                         void* pUserData = nullptr);

/**
 * @brief Устанавливает глобальные настройки библиотеки.
 *
 * @param[in] pOptions Опции библиотеки.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup init
 */
ILR_API ilr_status ILR_CALL ilr_set_options(const ilr_options* pOptions);

/**
 * @brief Возвращает глобальные настройки библиотеки.
 *
 * @param[out] pOptions Опции библиотеки.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup init
 */
ILR_API ilr_status ILR_CALL ilr_get_options(ilr_options* pOptions);

/**
 * @brief Создаёт дескриптор поиска считывателей.
 *
 * @param[out] pSearch Дескриптор поиска считывателей.
 *
 * @remark Эта функция создаёт поток поиска считывателей, если ещё не создан (один поток на
 * библиотеку). Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * считывателей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_get_search(ilr_handle* pSearch);

/**
 * @brief Создаёт дескриптор считывателя.
 *
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[out] pReader     Дескриптор считывателя.
 *
 * @remark Эта функция создаёт поток считывателя, если ещё не создан (один поток на порт).
 * Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 * Функция позволяет создать более одного дескриптора для одного порта.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_get_reader(ilr_port_type nPortType, const char* pszPortName,
                                           ilr_handle* pReader);

/**
 * @brief Устанавливает функцию обратного вызова для уведомлений поиска считывателей.
 *
 * Устанавливает функцию для получения сообщений от дескриптора поиска считывателей.
 *
 * @param[in] hSearch   Дескриптор поиска считывателей.
 * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать при возникновении
 * события поиска считывателей.
 * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в try catch.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе будет ошибка `ILR_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_set_message_callback(ilr_handle hSearch,
                                                            ilr_search_message_callback pCallback,
                                                            void* pUserData = nullptr);

/**
 * @brief Включает/выключает очередь сообщений.
 *
 * Эта функция устанавливает/снимает флаг "очередь включена" в дескрипторе поиска.
 * Очередь сообщений предназначена для синхронизации обработки сообщений поиска.
 *
 * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
 * вызывается функция обратного вызова, установленная функцией @ref ilr_search_set_message_callback,
 * из которой посылается сигнал потоку, обрабатывающему сообщения, этот поток при получении сигнала
 * циклично вызывает @ref ilr_search_get_message, чтобы получить и обработать все сообщения.
 *
 * @param[in] hSearch  Дескриптор поиска считывателей.
 * @param[in] fEnable  `ILR_TRUE`, включает очередь, иначе - выключает.
 *
 * @warning Если не извлекать сообщения из очереди функцией @ref ilr_search_get_message, то
 * она будет расти пока не закончится память.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_enable_message_queue(ilr_handle hSearch,
                                                            ilr_bool fEnable = ILR_TRUE);

/**
 * @brief Извлекает следующее сообщение из очереди.
 *
 * @param[in]  hSearch   Дескриптор поиска считывателей.
 * @param[out] pMsg      Тип сообщения.
 * @param[out] pMsgData  Указатель на данные пользователя, которые были установлены функцией
 * @ref ilr_search_set_message_callback.
 * @param[out] pFound    `ILR_TRUE` если сообщение успешно извлечено, иначе - очередь пуста.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_get_message(ilr_handle hSearch, ilr_search_msg* pMsg,
                                                   const void** pMsgData, ilr_bool* pFound);

/**
 * @brief Устанавливает параметры поиска считывателей.
 *
 * @remark Параметры поиска считывателей общие для всех дескрипторов.
 * Параметры не теряются при закрытии всех дескрипторов поиска.
 *
 * @param[in] hSearch   Дескриптор поиска считывателей.
 * @param[in] pOptions  Опции поиска считывателей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_set_options(ilr_handle hSearch,
                                                   const ilr_search_options* pOptions);

/**
 * @brief Возвращает параметры поиска считывателей.
 *
 * @param[in]  hSearch   Дескриптор поиска считывателей.
 * @param[out] pOptions  Опции поиска считывателей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_get_options(ilr_handle hSearch,
                                                   ilr_search_options* pOptions);

/**
 * @brief Устанавливает список портов для прослушки конвертеров к режиме "Клиент".
 *
 * Устанавливает список TCP-портов, к которым будут подключаться IP-конвертеры
 * в режиме "Клиент".
 *
 * @remark Список портов общий для всех дескрипторов.
 *
 * @param[in] hSearch Дескриптор поиска считывателей.
 * @param[in] pPorts  Массив портов. Если =nullptr, то очищает список.
 * @param[in] nCount  Количество элементов массива портов.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_set_listen_ports(ilr_handle hSearch, const uint16_t* pPorts,
                                                        size_t nCount);

/**
 * @brief Возвращает список портов для прослушки конвертеров к режиме "Клиент".
 *
 * Возвращает список TCP-портов, который был установлен функцией @ref ilr_search_set_listen_ports.
 *
 * @remark Список портов общий для всех дескрипторов.
 *
 * @param[in]     hSearch Дескриптор поиска считывателей.
 * @param[in out] pBuf    Указатель на буфер для списка портов. Может быть =nullptr.
 * @param[in out] pSize   Размер буфера в словах `uint16_t` или @ref ILR_AUTOALLOCATE. Если
 * =`ILR_AUTOALLOCATE`, то @p pBuf должен ссылаться на указатель, т.е. должен иметь тип uint16_t **,
 * функция сама выделит память под список портов, которую после использования нужно освободить
 * функцией @ref ilr_free_memory. В конце функция устанавливает в @p pSize требуемый размер для
 * буфера.
 *
 * @retval ILR_OK Функция выполнена успешно.
 * @retval ILR_E_BUFFER_TOO_SMALL Размер буфера не достаточный (когда не используется
 * `ILR_AUTOALLOCATE`).
 * @retval ILR_E_OUTOFMEMORY Не удалось выделить память (когда используется `ILR_AUTOALLOCATE`).
 * @retval ILR_E_POINTER Необходимый выходной параметр = nullptr.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_get_listen_ports(ilr_handle hSearch, uint16_t* pBuf,
                                                        size_t* pSize);

/**
 * @brief Возвращает состояние Tcp-порта, открытого для прослушки конвертеров в режиме "Клиент".
 *
 * @param[in]  hSearch   Дескриптор поиска конвертеров.
 * @param[in]  nTcpPort  Номер TCP порта.
 * @param[out] pStatus   Состояние порта. `ILR_OK` порт открыт успешно, `ILR_E_PENDING` порт в
 * процессе открытия или его нет в списке портов для прослушки, иначе не удалось открыть.
 *
 * @remark Чтобы открыть порт для прослушки нужно добавить его в список портов с помощью функции
 * @ref ilr_search_set_listen_ports, и включить поиск конвертеров в режиме "Клиент" с помощью
 * @ref ilr_search_set_options (установить pOptions->nReaderTypes |= ILR_READER_CLIENT).
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILR_E_POINTER Необходимый выходной параметр равен `nullptr`.
 */
ILR_API ilr_status ILR_CALL ilr_search_get_listen_status(ilr_handle hSearch, uint16_t nTcpPort,
                                                         ilr_status* pStatus);

/**
 * @brief Ищет считыватели.
 *
 * @param[in] hSearch  Дескриптор поиска считывателей.
 * @param[in] fReset   `ILR_TRUE`, очищает список найденных считывателей перед началом поиска.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * считывателей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_scan(ilr_handle hSearch, ilr_bool fReset = ILR_FALSE);

/**
 * @brief Запускает асинхронную команду поиска считывателей.
 *
 * @param[in]  hSearch   Дескриптор поиска считывателей.
 * @param[in]  fReset    `ILR_TRUE`, очистить список найденных перед поиском.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_begin_scan(ilr_handle hSearch, ilr_bool fReset,
                                                  ilr_handle* pCommand);

/**
 * @brief Возвращает количество найденных считывателей.
 *
 * @param[in]  hSearch Дескриптор поиска считывателей.
 * @param[out] pCount  Количество найденных считывателей.
 *
 * @warning Эта функция копирует список из потока поиска, поэтому её лучше не вызывать в цикле, а
 * сохранить значение @p *pCount в переменной, и уже её использовать в цикле.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_get_reader_count(ilr_handle hSearch, size_t* pCount);

/**
 * @brief Возвращает информацию о найденном считывателе.
 *
 * @param[in]  hSearch Дескриптор поиска считывателей.
 * @param[in]  nIdx    Позиция в списке найденных считывателей.
 * @param[out] pInfo   Информация о считывателе.
 *
 * @warning В @p pInfo ссылки `pszPortName` и `pszConnect` действительны до следующего вызова @ref
 * ilr_search_get_reader_count и пока жив дескриптор @p hSearch.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_get_reader_info(ilr_handle hSearch, size_t nIdx,
                                                       ilr_reader_info* pInfo);

/**
 * @brief Включает/выключает авто поиск считывателей.
 *
 * @param[in] hSearch  Дескриптор поиска считывателей.
 * @param[in] fEnable  `ILR_TRUE`, включает авто поиск, иначе - выключает.
 * @param[in] fWait    `ILR_TRUE`, ждёт полного завершения команды, иначе только устанавливает
 * флаг.
 *
 * @remark Если @p fWait равен `ILR_TRUE`, то функция не возвращает управление пока ждёт выполнение
 * команды в потоке поиска считывателей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_set_auto_scan(ilr_handle hSearch,
                                                     ilr_bool fEnable = ILR_TRUE,
                                                     ilr_bool fWait = ILR_TRUE);

/**
 * @brief Запускает асинхронную команду вкл/выкл режим авто поиска считывателей.
 *
 * @param[in]  hSearch   Дескриптор поиска считывателей.
 * @param[in]  fEnable   `ILR_TRUE`, включает поиск в реальном времени, иначе - выключает.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_begin_set_auto_scan(ilr_handle hSearch, ilr_bool fEnable,
                                                           ilr_handle* pCommand);

/**
 * @brief Возвращает флаг авто поиска считывателей.
 *
 * @param[in]  hSearch   Дескриптор поиска считывателей.
 * @param[out] pEnabled  `ILR_TRUE`, авто поиск включен, иначе - выключен.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILR_API ilr_status ILR_CALL ilr_search_get_auto_scan(ilr_handle hSearch, ilr_bool* pEnabled);

/**
 * @brief Открывает порт и возвращает его дескриптор.
 *
 * @param[in]  hSearch     Дескриптор поиска считывателей.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[out] pInfo       Информация о считывателе.
 * @param[out] pPort       Системный дескриптор порта.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * считывателей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_search_open_port(ilr_handle hSearch, ilr_port_type nPortType,
                                                 const char* pszPortName, ilr_reader_info* pInfo,
                                                 int* pPort);

/**
 * @brief Запускает асинхронную команду открытия порта.
 *
 * @param[in]  hSearch     Дескриптор поиска считывателей.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_search_begin_open_port(ilr_handle hSearch, ilr_port_type nPortType,
                                                       const char* pszPortName,
                                                       ilr_handle* pCommand);

/**
 * @brief Возвращает результат открытия порта.
 *
 * @param[in]  hCommand Дескриптор команды.
 * @param[out] pPortFD  Системный дескриптор порта.
 * @param[out] pInfo    Информация о конвертере (если известно).
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_search_end_open_port(ilr_handle hCommand, int* pPortFD,
                                                     ilr_reader_info* pInfo);

/**
 * @brief Закрывает порт.
 *
 * @param[in]  hSearch     Дескриптор поиска считывателей.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[in]  hPort       Системный дескриптор порта.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * считывателей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_search_close_port(ilr_handle hSearch, ilr_port_type nPortType,
                                                  const char* pszPortName, int hPort);

/**
 * @brief Запускает асинхронную команду закрытия порта.
 *
 * @param[in]  hSearch     Дескриптор поиска считывателей.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[in]  hPortFD     Системный дескриптор порта.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_search_begin_close_port(ilr_handle hSearch, ilr_port_type nPortType,
                                                        const char* pszPortName, int hPortFD,
                                                        ilr_handle* pCommand);

/**
 * @brief Устанавливает функцию обратного вызова для уведомлений считывателя.
 *
 * Устанавливает функцию для получения сообщений от дескриптора считывателя.
 *
 * @param[in] hReader   Дескриптор считывателя.
 * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать при возникновении
 * события считывателя.
 * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе вернёт ошибку `ILR_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_set_message_callback(ilr_handle hReader,
                                                            ilr_reader_message_callback pCallback,
                                                            void* pUserData = nullptr);

/**
 * @brief Включает/выключает очередь сообщений.
 *
 * Эта функция устанавливает/снимает флаг в дескрипторе считывателя.
 * Очередь сообщений предназначена для синхронизации обработки сообщений.
 *
 * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
 * вызывается функция обратного вызова, установленная функцией @ref ilr_reader_set_message_callback,
 * из которой посылается сигнал потоку, обрабатывающему сообщения, этот поток при получении сигнала
 * циклично вызывает @ref ilr_reader_get_message, чтобы получить и обработать все сообщения.
 *
 * @param[in] hReader  Дескриптор считывателя.
 * @param[in] fEnable  `ILR_TRUE`, включает очередь, иначе - выключает.
 *
 * @warning Если не извлекать сообщения из очереди функцией @ref ilr_reader_get_message, то
 * она будет расти пока не закончится память.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_enable_message_queue(ilr_handle hReader,
                                                            ilr_bool fEnable = ILR_TRUE);

/**
 * @brief Извлекает следующее сообщение из очереди.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[out] pMsg      Тип сообщения.
 * @param[out] pMsgData  Указатель на данные пользователя, которые были установлены функцией
 * @ref ilr_reader_set_message_callback.
 * @param[out] pFound    `ILR_TRUE` если сообщение успешно извлечено, иначе - очередь пуста.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_get_message(ilr_handle hReader, ilr_reader_msg* pMsg,
                                                   const void** pMsgData, ilr_bool* pFound);

/**
 * @brief Устанавливает параметры считывателя.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[in]  pOptions Параметры считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_set_options(ilr_handle hReader,
                                                   const ilr_reader_options* pOptions);

/**
 * @brief Возвращает параметры считывателя.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[in]  pOptions Параметры считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_get_options(ilr_handle hReader,
                                                   ilr_reader_options* pOptions);

/**
 * @brief Подключается к считывателю.
 *
 * @param[in] hReader     Дескриптор считывателя.
 * @param[in] fReconnect `ILR_TRUE` Отключается перед подключением.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_connect(ilr_handle hReader, ilr_bool fReconnect = ILR_FALSE);

/**
 * @brief Запускает асинхронную команду подключения к считывателю.
 *
 * @param[in]  hReader     Дескриптор считывателя.
 * @param[in]  fReconnect  `ILR_TRUE` переподключиться.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_connect(ilr_handle hReader, ilr_bool fReconnect,
                                                     ilr_handle* pCommand);

/**
 * @brief Отключается от считывателя.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_disconnect(ilr_handle hReader);

/**
 * @brief Запускает асинхронную команду отключения от считывателя.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_disconnect(ilr_handle hReader, ilr_handle* pCommand);

/**
 * @brief Возвращает состояние подключения к считывателю.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[out] pStatus  Состояние подключения к считывателю.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_get_connection_status(ilr_handle hReader,
                                                             ilr_connection_status* pStatus);

/**
 * @brief Возвращает информацию о считывателе.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[out] pInfo    Информация о считывателе.
 *
 * @warning В @p pInfo ссылки `pszPortName` и `pszConnect` действительны пока жив дескриптор @p
 * hReader.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_get_info(ilr_handle hReader, ilr_reader_info* pInfo);

/**
 * @brief Ищет карту в поле считывателя.
 *
 * @param[in] hReader    Дескриптор считывателя.
 * @param[in] fReset     `ILR_TRUE`, сбросить старые результаты поиска.
 * @param[in] fPowerOff  `ILR_TRUE`, выключает RF поле после сканирования.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_scan(ilr_handle hReader, ilr_bool fReset = ILR_FALSE,
                                            ilr_bool fPowerOff = ILR_TRUE);

/**
 * @brief Запускает асинхронную команду поиска карты.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  fReset     `ILR_TRUE`, сбросить старые результаты поиска.
 * @param[in]  fPowerOff  `ILR_TRUE`, выключает RF поле после сканирования.
 * @param[out] pCommand   Дескриптор команды.
 *
 * @return `ILR_OK `в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_scan(ilr_handle hReader, ilr_bool fReset,
                                                  ilr_bool fPowerOff, ilr_handle* pCommand);

/**
 * @brief Возвращает информацию о карте в поле считывателя.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[out] pInfo    Информация о карте. Если pInfo.rUID.nLength = 0, то нет карты.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_get_card_info(ilr_handle hReader, ilr_card_info* pInfo);

/**
 * @brief Включает/выключает автоматическое сканирование карт.
 *
 * Если включено, то ищет карты в фоновом режиме.
 *
 * @param[in] hReader  Дескриптор считывателя.
 * @param[in] fEnable  `ILR_TRUE`, включить авто сканирование, иначе - выключить.
 * @param[in] fWait    `ILR_TRUE`, ждать полного завершения команды, иначе только установить флаг.
 *
 * @remark Если @p fWait равен `ILR_TRUE`, то функция не возвращает управление пока ждёт
 * выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_set_auto_scan(ilr_handle hReader,
                                                     ilr_bool fEnable = ILR_TRUE,
                                                     ilr_bool fWait = ILR_TRUE);

/**
 * @brief Запускает асинхронную команду вкл/выкл автоматического сканирования карт.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  fEnable    `ILR_TRUE`, включает авто сканирование, иначе - выключает.
 * @param[out] pCommand   Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_set_auto_scan(ilr_handle hReader, ilr_bool fEnable,
                                                           ilr_handle* pCommand);

/**
 * @brief Возвращает флаг автоматическое сканирование карт.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[out] pEnabled  `ILR_TRUE`, если авто сканирование включено.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_get_auto_scan(ilr_handle hReader, ilr_bool* pEnabled);

/**
 * @brief Читает данные карты Mifare Ultralight.
 *
 * @param[in]  hReader     Дескриптор считывателя.
 * @param[in]  nPageIdx    Номер первой читаемой страницы (0..15).
 * @param[out] pBuf        Буфер для прочитанных страниц.
 * @param[in]  nPageCount  Количество страниц, которые нужно прочитать.
 * @param[out] pRead       Количество прочитанных страниц. Может быть равен `nullptr`.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_read_mf_ultralight(ilr_handle hReader, size_t nPageIdx,
                                                          uint32_t* pBuf, size_t nPageCount,
                                                          size_t* pRead = nullptr);

/**
 * @brief Запускает асинхронную команду чтения данных из карты Mifare Ultralight.
 *
 * @param[in]  hReader     Дескриптор считывателя.
 * @param[in]  nPageIdx    Номер первой читаемой страницы (начиная от 0).
 * @param[in]  nPageCount  Количество читаемых страниц.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_read_mf_ultralight(ilr_handle hReader, size_t nPageIdx,
                                                                size_t nPageCount,
                                                                ilr_handle* pCommand);

/**
 * @brief Возвращает результат чтения данных из карты Mifare Ultralight.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] ppList    Ссылка на список прочитанных страниц. Ссылка действительна до закрытия
 * дескриптора команды @p hCommand.
 * @param[out] pRead     Количество прочитанных страниц.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_read_mf_ultralight(ilr_handle hCommand,
                                                              const uint32_t** ppList,
                                                              size_t* pRead);

/**
 * @brief Пишет данные карты Mifare Ultralight.
 *
 * @param[in]  hReader     Дескриптор считывателя.
 * @param[in]  nPageIdx    Номер первой записываемой страницы (0..15).
 * @param[in]  pData       Данные страниц.
 * @param[in]  nPageCount  Количество страниц, которые нужно записать.
 * @param[out] pWritten    Количество записанных страниц. Может быть равен `nullptr`.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_write_mf_ultralight(ilr_handle hReader, size_t nPageIdx,
                                                           const uint32_t* pData, size_t nPageCount,
                                                           size_t* pWritten = nullptr);

/**
 * @brief Запускает асинхронную команду записи данных в карту Mifare Ultralight.
 *
 * @param[in]  hReader     Дескриптор считывателя.
 * @param[in]  nPageIdx    Номер первой записываемой страницы (начиная от 0).
 * @param[in]  pData       Данные страниц.
 * @param[in]  nPageCount  Количество записываемых страниц.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_write_mf_ultralight(ilr_handle hReader,
                                                                 size_t nPageIdx,
                                                                 const uint32_t* pData,
                                                                 size_t nPageCount,
                                                                 ilr_handle* pCommand);

/**
 * @brief Возвращает результат записи данных в карту Mifare Ultralight.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pWritten  Количество прочитанных страниц.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_write_mf_ultralight(ilr_handle hCommand,
                                                               size_t* pWritten);

/**
 * @brief Загружает ключ для авторизации сектора Mifare Classic / Plus SL1.
 *
 * @param[in] hReader  Дескриптор считывателя.
 * @param[in] nKey     Ключ аутентификации `Mifare Classic`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_load_mf_ckey(ilr_handle hReader,
                                                    const ilr_mf_classic_key& nKey);

/**
 * @brief Загружает ключ для авторизации сектора Mifare Plus SL3.
 *
 * @param[in] hReader  Дескриптор считывателя.
 * @param[in] rKey     Ключ аутентификации `Mifare Plus`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_load_mf_pkey(ilr_handle hReader,
                                                    const ilr_mf_plus_key& rKey);

/**
 * @brief Авторизует сектор карты Mifare Classic/Plus.
 *
 * Авторизует сектор карты Mifare Classic/Plus, используя ключ, загруженный
 * функцией @ref ilr_reader_load_mf_ckey / @ref ilr_reader_load_mf_pkey.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[in]  nAddress  Номер блока (0..255) или адрес `Mifare Plus`.
 * @param[in]  fKeyB     `ILR_TRUE`, авторизовать по ключу `Б`, иначе - по ключу `А`.
 * @param[out] pAuthOk   `ILR_TRUE`, сектор успешно авторизован.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_auth_mf_card(ilr_handle hReader, size_t nAddress,
                                                    ilr_bool fKeyB, ilr_bool* pAuthOk);

/**
 * @brief Запускает асинхронную команду авторизации сектора карты Mifare Classic / Plus
 * используя ключ, загруженный функцией @ref ilr_reader_load_mf_ckey / @ref ilr_reader_load_mf_pkey.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[in]  nAddress  Номер блока или адрес `Mifare Plus`.
 * @param[in]  fKeyB     `ILR_TRUE`, авторизовать по ключу `Б`, иначе `А`.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_auth_mf_card(ilr_handle hReader, size_t nAddress,
                                                          ilr_bool fKeyB, ilr_handle* pCommand);

/**
 * @brief Возвращает результат авторизации сектора карты.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pAuthOk   `ILR_TRUE`, авторизация прошла успешно.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_auth_mf_card(ilr_handle hCommand, ilr_bool* pAuthOk);

/**
 * @brief Авторизует сектор карты Mifare Classic / Plus, используя ключи считывателя.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nAddress   Номер блока (0..255) или адрес `Mifare Plus`.
 * @param[in]  fKeyB      `ILR_TRUE`, авторизовать по ключу `Б`, иначе - по ключу `А`.
 * @param[in]  nKeys      Биты (0..15) ключей в памяти считывателя.
 * @param[out] pFoundIdx  Позиция найденного ключа в памяти считывателя. Если -1, ключ не найден.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_auth_mf_card2(ilr_handle hReader, size_t nAddress,
                                                     ilr_bool fKeyB, uint32_t nKeys = 0xFFFF,
                                                     int* pFoundIdx = nullptr);

/**
 * @brief Запускает асинхронную команду авторизации сектора карты Mifare Classic / Plus.
 * используя ключи в памяти считывателя
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[in]  nAddress  Номер блока или адрес `Mifare Plus`,
 * @param[in]  fKeyB     `ILR_TRUE`, авторизовать по ключу `Б`, иначе `А`.
 * @param[in]  nKeys     Биты номеров ключей в памяти считывателя.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_auth_mf_card2(ilr_handle hReader, size_t nAddress,
                                                           ilr_bool fKeyB, uint32_t nKeys,
                                                           ilr_handle* pCommand);

/**
 * @brief Возвращает результат авторизации сектора карты.
 *
 * @param[in]  hCommand   Дескриптор команды.
 * @param[out] pFoundIdx  Позиция найденного ключа в памяти считывателя. Если -1, ключ не найден.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_auth_mf_card2(ilr_handle hCommand, int* pFoundIdx);

/**
 * @brief Читает данные карты Mifare Classic или Mifare Plus SL1.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер первого читаемого блока (0..255).
 * @param[out] pBuf         Буфер для прочитанных блоков.
 * @param[out] nBlockCount  Количество блоков, которые нужно прочитать.
 * @param[in]  pRead        Количество прочитанных блоков. Может быть равен `nullptr`.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_read_mf_classic(ilr_handle hReader, size_t nBlockIdx,
                                                       ilr_mf_block_data* pBuf, size_t nBlockCount,
                                                       size_t* pRead = nullptr);

/**
 * @brief Запускает асинхронную команду чтения данных из карты Mifare Classic или Mifare Plus SL1.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер блока `Mifare Classic`.
 * @param[in]  nBlockCount  Количество блоков.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_read_mf_classic(ilr_handle hReader, size_t nBlockIdx,
                                                             size_t nBlockCount,
                                                             ilr_handle* pCommand);

/**
 * @brief Возвращает результат чтения данных из карты Mifare Classic или Mifare Plus SL1.
 *
 * @param[in]  hCommand   Дескриптор команды.
 * @param[out] ppList     Ссылка на список прочитанных блоков. Ссылка действительна до закрытия
 * дескриптора команды @p hCommand.
 * @param[out] pRead      Количество прочитанных блоков.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_read_mf_classic(ilr_handle hCommand,
                                                           const ilr_mf_block_data** ppList,
                                                           size_t* pRead);

/**
 * @brief Пишет данные карты Mifare Classic или Mifare Plus SL1.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер первого записываемого блока (0..255).
 * @param[in]  pData        Данные записываемых блоков.
 * @param[out] nBlockCount  Количество блоков, которые нужно записать.
 * @param[out] pWritten     Количество записанных блоков. Может быть равен `nullptr`.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_write_mf_classic(ilr_handle hReader, size_t nBlockIdx,
                                                        const ilr_mf_block_data* pData,
                                                        size_t nBlockCount,
                                                        size_t* pWritten = nullptr);

/**
 * @brief Запускает асинхронную команду записи данных в карту Mifare Classic или Mifare Plus SL1.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер блока `Mifare Classic`.
 * @param[in]  pData        Данные блоков для записи.
 * @param[in]  nBlockCount  Количество записываемых блоков.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_write_mf_classic(ilr_handle hReader, size_t nBlockIdx,
                                                              const ilr_mf_block_data* pData,
                                                              size_t nBlockCount,
                                                              ilr_handle* pCommand);

/**
 * @brief Возвращает результат записи данных в карту Mifare Classic или Mifare Plus SL1.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pWritten  Количество записанных блоков.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_write_mf_classic(ilr_handle hCommand, size_t* pWritten);

/**
 * @brief Читает данные карты Mifare Plus SL3.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nAddress     Номер первого читаемого блока (0..255).
 * @param[out] pBuf         Буфер для прочитанных блоков.
 * @param[in]  nBlockCount  Количество блоков, которые нужно прочитать.
 * @param[in]  fOpenText    `ILR_TRUE`, открытая передача, иначе - зашифрованная.
 * @param[out] pRead        Количество прочитанных блоков. Может быть равен `nullptr`.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_read_mf_plus(ilr_handle hReader, size_t nAddress,
                                                    ilr_mf_block_data* pBuf, size_t nBlockCount,
                                                    ilr_bool fOpenText = ILR_TRUE,
                                                    size_t* pRead = nullptr);

/**
 * @brief Запускает асинхронную команду чтения данных из карты Mifare Plus SL3.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nAddress     Номер блока карты Mifare.
 * @param[in]  nBlockCount  Количество блоков, которые нужно прочитать.
 * @param[in]  fOpenText    `ILR_TRUE`, открытая передача, иначе зашифрованная.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_read_mf_plus(ilr_handle hReader, size_t nAddress,
                                                          size_t nBlockCount, ilr_bool fOpenText,
                                                          ilr_handle* pCommand);

/**
 * @brief Возвращает результат чтения данных из карты Mifare Plus SL3.
 *
 * @param[in]  hCommand   Дескриптор команды.
 * @param[out] ppList     Ссылка на список прочитанных блоков. Ссылка действительна до закрытия
 * дескриптора команды @p hCommand.
 * @param[out] pRead      Количество прочитанных блоков. Может быть равно `nullptr`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_read_mf_plus(ilr_handle hCommand,
                                                        const ilr_mf_block_data** ppList,
                                                        size_t* pRead);

/**
 * @brief Пишет данные карты Mifare Plus SL3.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nAddress     Номер первого записываемого блока (0..255) или адрес `Mifare Plus`.
 * @param[in]  pData        Данные записываемых блоков.
 * @param[in]  nBlockCount  Количество блоков, которые нужно записать.
 * @param[in]  fOpenText    `ILR_TRUE`, открытая передача, иначе - зашифрованная.
 * @param[out] pWritten     Количество записанных блоков. Может быть равно `nullptr`.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_write_mf_plus(ilr_handle hReader, size_t nAddress,
                                                     const ilr_mf_block_data* pData,
                                                     size_t nBlockCount,
                                                     ilr_bool fOpenText = ILR_TRUE,
                                                     size_t* pWritten = nullptr);

/**
 * @brief Запускает асинхронную команду записи данных в карту Mifare Plus SL3.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nAddress     Номер блока или адрес `Mifare Plus`.
 * @param[in]  pData        Данные блоков для записи.
 * @param[in]  nBlockCount  Количество блоков.
 * @param[in]  fOpenText    `ILR_TRUE`, открытая передача, иначе зашифрованная.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_write_mf_plus(ilr_handle hReader, size_t nAddress,
                                                           const ilr_mf_block_data* pData,
                                                           size_t nBlockCount, ilr_bool fOpenText,
                                                           ilr_handle* pCommand);

/**
 * @brief Возвращает результат записи данных в карту Mifare Plus SL3.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pWritten  Количество записанных блоков.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_write_mf_plus(ilr_handle hCommand, size_t* pWritten);

/**
 * @brief Увеличивает содержимое блока-значения карты Mifare и сохраняет результат во временном
 * регистре данных.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nBlockIdx  Номер блока (0..255).
 * @param[in]  nValue     Величина инкремента.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_increment(ilr_handle hReader, size_t nBlockIdx,
                                                    uint32_t nValue);

/**
 * @brief Увеличивает содержимое блока-значения карты Mifare и сохраняет результат во временном
 * регистре данных.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nBlockIdx  Номер блока `Mifare Classic/Plus`.
 * @param[in]  nValue     Величина инкремента.
 * @param[out] pCommand   Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_mf_increment(ilr_handle hReader, size_t nBlockIdx,
                                                          uint32_t nValue, ilr_handle* pCommand);

/**
 * @brief Уменьшает содержимое блока-значения карты Mifare и сохраняет результат во временном
 * регистре данных.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nBlockIdx  Номер блока (0..255).
 * @param[in]  nValue     Величина декремента.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_decrement(ilr_handle hReader, size_t nBlockIdx,
                                                    uint32_t nValue);

/**
 * @brief Уменьшает содержимое блока-значения карты Mifare и сохраняет результат во временном
 * регистре данных.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nBlockIdx  Номер блока `Mifare Classic/Plus`.
 * @param[in]  nValue     Величина инкремента.
 * @param[out] pCommand   Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_mf_decrement(ilr_handle hReader, size_t nBlockIdx,
                                                          uint32_t nValue, ilr_handle* pCommand);

/**
 * @brief Записывает содержимое во временном регистре данных в блок-значение.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nBlockIdx  Номер блока (0..255).
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_transfer(ilr_handle hReader, size_t nBlockIdx);

/**
 * @brief Записывает содержимое во временном регистре данных в блок-значение.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nBlockIdx  Номер блока `Mifare Classic/Plus`.
 * @param[out] pCommand   Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_mf_transfer(ilr_handle hReader, size_t nBlockIdx,
                                                         ilr_handle* pCommand);

/**
 * @brief Перемещает содержимое блока в регистр данных Mifare.
 *
 * @param[in] hReader    Дескриптор считывателя.
 * @param[in] nBlockIdx  Номер блока (0..255).
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_restore(ilr_handle hReader, size_t nBlockIdx);

/**
 * @brief Перемещает содержимое блока в регистр данных Mifare.
 *
 * @param[in]  hReader    Дескриптор считывателя.
 * @param[in]  nBlockIdx  Номер блока `Mifare Classic/Plus`.
 * @param[out] pCommand   Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_mf_restore(ilr_handle hReader, size_t nBlockIdx,
                                                        ilr_handle* pCommand);

/**
 * @brief Выключает RF поле считывателя (после выключения нужно подождать 10 мс).
 *
 * @param[in]  hReader  Дескриптор считывателя.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_power_off(ilr_handle hReader);

/**
 * @brief R+A+S(Request+Anticollision+Select)
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[in]  fWakeUp  `ILR_TRUE`, разбудить карту.
 * @param[out] pSAK     Код `SAK` карты.
 * @param[out] pATQ     Код `ATQ` карты.
 * @param[out] pUID     Номер карты.
 * @param[out] pFound   `ILR_TRUE`, карта найдена, иначе нет карты.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_ras(ilr_handle hReader, ilr_bool fWakeUp, uint8_t* pSAK,
                                              uint16_t* pATQ, ilr_card_uid* pUID, ilr_bool* pFound);

/**
 * @brief R+R(Request+Reselect(по известному номеру)).
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[in]  fWakeUp  `ILR_TRUE`, разбудить карту.
 * @param[in]  rUID     Номер карты.
 * @param[out] pFound	`ILR_TRUE`, карта найдена.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_rr(ilr_handle hReader, ilr_bool fWakeUp,
                                             const ilr_card_uid& rUID, ilr_bool* pFound);

/**
 * @brief Halt.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_halt(ilr_handle hReader);

/**
 * @brief Переходит на ISO 14443-4.
 *
 * @param[in]     hReader  Дескриптор считывателя.
 * @param[out]    pAtsBuf  Буфер для данных ATS.
 * @param[in out] pSize    Размер буфера. Обычно нужно 12 байт.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_rats(ilr_handle hReader, uint8_t* pAtsBuf = nullptr,
                                               size_t* pSize = nullptr);

/**
 * @brief Записывает ключи AES и всех блоков.
 *
 * @param[in] hReader   Дескриптор считывателя.
 * @param[in] nAddress  Адрес ключа.
 * @param[in] rKey      Значение ключа.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_write_perso(ilr_handle hReader, uint32_t nAddress,
                                                      const ilr_mf_plus_key& rKey);

/**
 * @brief Переключает Mifare Plus в SL1 или SL3(если SL1 нет).
 *
 * @param[in] hReader  Дескриптор считывателя.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_mf_commit_perso(ilr_handle hReader);

/**
 * @brief Записывает ключи аутентификации Mifare Classic в память считывателя.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[in]  nIdx      Номер ячейки в памяти считывателя, в которую записывается первый ключ.
 * @param[in]  fKeyB     `ILR_TRUE`, ключ `Б`, иначе - ключ `А`.
 * @param[in]  pKeys     Список записываемых ключей.
 * @param[in]  nCount    Количество ключей, которые нужно записать.
 * @param[out] pWritten  Количество записанных ключей.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_write_mf_ckey(ilr_handle hReader, size_t nIdx,
                                                     ilr_bool fKeyB,
                                                     const ilr_mf_classic_key* pKeys, size_t nCount,
                                                     size_t* pWritten = nullptr);

/**
 * @brief Запускает асинхронную команду записи ключей аутентификации Mifare Classic в память
 * считывателя.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[in]  nIdx      Позиция первого ключа в памяти считывателя.
 * @param[in]  fKeyB     `ILR_TRUE`, ключ `B`, иначе ключ `A`.
 * @param[in]  pKeys     Список ключей аутентификации.
 * @param[in]  nCount    Количество ключей аутентификации.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_write_mf_ckey(ilr_handle hReader, size_t nIdx,
                                                           ilr_bool fKeyB,
                                                           const ilr_mf_classic_key* pKeys,
                                                           size_t nCount, ilr_handle* pCommand);

/**
 * @brief Возвращает результат записи ключей аутентификации Mifare Classic в память считывателя.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pWritten  Количество записанных ключей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_write_mf_ckey(ilr_handle hCommand, size_t* pWritten);

/**
 * @brief Записывает ключи аутентификации Mifare Plus в память считывателя
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[in]  nIdx      Номер ячейки в памяти считывателя, в которую записывается первый ключ.
 * @param[in]  fKeyB     `ILR_TRUE`, ключ `Б`, иначе - ключ `А`.
 * @param[in]  pKeys     Список записываемых ключей.
 * @param[in]  nCount    Количество ключей, которые нужно записать.
 * @param[out] pWritten  Количество записанных ключей.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_write_mf_pkey(ilr_handle hReader, size_t nIdx,
                                                     ilr_bool fKeyB, const ilr_mf_plus_key* pKeys,
                                                     size_t nCount, size_t* pWritten = nullptr);

/**
 * @brief Запускает асинхронную команду записи ключей аутентификации Mifare Plus в память
 * считывателя.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[in]  nIdx      Позиция первого ключа в памяти считывателя.
 * @param[in]  fKeyB     `ILR_TRUE`, ключ B, иначе ключ A.
 * @param[in]  pKeys     Список ключей аутентификации.
 * @param[in]  nCount    Количество ключей аутентификации.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_write_mf_pkey(ilr_handle hReader, size_t nIdx,
                                                           ilr_bool fKeyB,
                                                           const ilr_mf_plus_key* pKeys,
                                                           size_t nCount, ilr_handle* pCommand);

/**
 * @brief Возвращает результат записи ключей аутентификации Mifare Plus в память считывателя.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pWritten  Количество записанных ключей.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_write_mf_pkey(ilr_handle hCommand, size_t* pWritten);

/**
 * @brief Загружает пароль Temic в память объекта считывателя.
 *
 * @param[in] hReader    Дескриптор считывателя.
 * @param[in] nPassword  Пароль `Temic`. Если равно -1, то нет пароля.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_load_temic_password(ilr_handle hReader,
                                                           const int64_t& nPassword);

/**
 * @brief Ищет карту Temic в поле считывателя.
 *
 * @param[in] hReader     Дескриптор считывателя.
 * @param[in] nScanParam  Параметры сканирования `Temic`. Если -1, авто определение.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_scan_temic(ilr_handle hReader, int nScanParam = -1);

/**
 * @brief Запускает асинхронную команду поиска карты Temic в поле считывателя.
 *
 * @param[in]  hReader     Дескриптор считывателя.
 * @param[in]  nScanParam  Параметры сканирования `Temic`. Если -1, авто определение.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_scan_temic(ilr_handle hReader, int nScanParam,
                                                        ilr_handle* pCommand);

/**
 * @brief Вкл/выкл сканирование карт Temic (для Z-2 Rd-All и Z-2 EHR).
 *
 * Если включено, то ищет карту Temic при ручном или автоматическом поиске карт ( @ref
 * ilr_reader_scan, @ref ilr_reader_set_auto_scan).
 *
 * @param[in] hReader  Дескриптор считывателя.
 * @param[in] fEnable  `ILR_TRUE`, включает сканирование `Temic`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_set_scan_temic(ilr_handle hReader,
                                                      ilr_bool fEnable = ILR_TRUE);

/**
 * @brief Возвращает True если сканирование Temic включено.
 *
 * @param[in]  hReader  Дескриптор считывателя.
 * @param[out] pEnable  `ILR_TRUE`, если сканирование `Temic` включено.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_get_scan_temic(ilr_handle hReader, ilr_bool* pEnable);

/**
 * @brief Читает данные карты Temic.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер первого блока, который нужно причитать (0..9).
 * @param[out] pBuf         Буфер для прочитанных данных.
 * @param[in]  nBlockCount  Количество блоков, которые нужно прочитать.
 * @param[in]  nScanParam   Параметры сканирования `Temic`. Если -1, авто определение.
 * @param[out] pRead        Количество прочитанных блоков.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_read_temic(ilr_handle hReader, size_t nBlockIdx,
                                                  uint32_t* pBuf, size_t nBlockCount,
                                                  int nScanParam = -1, size_t* pRead = nullptr);

/**
 * @brief Запускает асинхронную команду чтения данных из карты Temic.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер блока карты.
 * @param[in]  nBlockCount  Количество блоков.
 * @param[in]  nScanParam   Параметр сканирования `Temic`. Если -1, авто определение.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_read_temic(ilr_handle hReader, size_t nBlockIdx,
                                                        size_t nBlockCount, int nScanParam,
                                                        ilr_handle* pCommand);

/**
 * @brief Возвращает результат чтения данных из карты Temic.
 *
 * @param[in]  hCommand   Дескриптор команды.
 * @param[out] ppList     Ссылка на список прочитанных блоков. Ссылка действительна до закрытия
 * дескриптора команды @p hCommand.
 * @param[out] pRead      Количество прочитанных блоков.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_read_temic(ilr_handle hCommand, const uint** ppList,
                                                      size_t* pRead);

/**
 * @brief Пишет данные карты Temic.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер первого блока, в который нужно записать (0..7).
 * @param[in]  pData        Данные блоков для записи.
 * @param[in]  nBlockCount  Количество блоков, которые нужно записать.
 * @param[in]  fLock        `ILR_TRUE`, заблокировать блоки от перезаписи.
 * @param[in]  nScanParam   Параметры сканирования `Temic`. Если -1, авто определение.
 * @param[out] pWritten     Количество записанных блоков.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_write_temic(ilr_handle hReader, size_t nBlockIdx,
                                                   const uint32_t* pData, size_t nBlockCount,
                                                   ilr_bool fLock = ILR_FALSE, int nScanParam = -1,
                                                   size_t* pWritten = nullptr);

/**
 * @brief Запускает асинхронную команду записи данных в карту Temic.
 *
 * @param[in]  hReader      Дескриптор считывателя.
 * @param[in]  nBlockIdx    Номер блока карты.
 * @param[in]  pData        Данные для записи.
 * @param[in]  nBlockCount  Количество блоков.
 * @param[in]  fLock        `ILR_TRUE`, блокировать от перезаписи.
 * @param[in]  nScanParam   Параметр сканирования `Temic`. Если -1, авто определение.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_write_temic(ilr_handle hReader, size_t nBlockIdx,
                                                         const uint32_t* pData, size_t nBlockCount,
                                                         ilr_bool fLock, int nScanParam,
                                                         ilr_handle* pCommand);

/**
 * @brief Возвращает результат записи данных в карту Temic.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pWritten  Количество записанных блоков.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_end_write_temic(ilr_handle hCommand, size_t* pWritten);

/**
 * @brief Сброс TRES.
 *
 * @param[in] hReader  Дескриптор считывателя.
 * @param[in] fWait    `ILR_TRUE`, ждать завершения команды.
 *
 * @remark Если @p fWait равно `ILR_TRUE`, то функция не возвращает управление пока ждёт выполнение
 * команды в потоке считывателя.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_reset_temic(ilr_handle hReader, ilr_bool fWait = ILR_TRUE);

/**
 * @brief Запускает асинхронную команду сброса TRES.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_reset_temic(ilr_handle hReader, ilr_handle* pCommand);

/**
 * @brief Запускает асинхронную команду сброса TRES.
 *
 * @param[in]  hReader   Дескриптор считывателя.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup reader
 */
ILR_API ilr_status ILR_CALL ilr_reader_begin_reset_temic(ilr_handle hReader, ilr_handle* pCommand);

/**
 * @brief Кодирует данные для эмуляции Em-Marine, для записи в блоки 0..2.
 *
 * @param[in]  rUID         Номер `Em-Marine`, который нужно эмулировать.
 * @param[out] pBuf3        Буфер для данных 3-х блоков.
 * @param[in]  nBlockCount  Размер буфера в блоках.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_encode_temic_emmarine(const ilr_card_uid& rUID, uint32_t* pBuf3,
                                                      size_t nBlockCount);

/**
 * @brief Декодирует номер Em-Marine из данных блоков 0..2 карты Temic.
 *
 * @param[in]  pData3       Данные блоков 0..2.
 * @param[in]  nBlockCount  Количество блоков. Должно быть не меньше 3.
 * @param[out] pUID         Номер `Em-Marine`. Если `Em-Marine` не обнаружен, то пустой номер.
 * @param[out] pConfigOk    `ILR_TRUE`, конфигурация `Temic` для эмуляции `Em-Marine` правильная.
 * Может быть равно `nullptr`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_decode_temic_emmarine(const uint32_t* pData3, size_t nBlockCount,
                                                      ilr_card_uid* pUID,
                                                      ilr_bool* pConfigOk = nullptr);

/**
 * @brief Кодирует данные для эмуляции HID, для записи в блоки 0..3.
 *
 * @param[in]  rUID         Номер `HID`, который нужно эмулировать.
 * @param[out] pBuf4        Буфер для данных 4-х блоков.
 * @param[in]  nBlockCount  Размер буфера в блоках.
 * @param[in]  nWiegand     Номер кодировки Wiegand 18..37.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_encode_temic_hid(const ilr_card_uid& rUID, uint32_t* pBuf4,
                                                 size_t nBlockCount, int nWiegand);

/**
 * @brief Декодирует номер HID из данных блоков 0..3 карты Temic.
 *
 * @param[in]  pData4       Данные блоков 0..3.
 * @param[in]  nBlockCount  Количество блоков. Должно быть не меньше 4.
 * @param[out] pUID         Номер `HID`. Если `HID` не обнаружен, то пустой номер.
 * @param[out] pWiegand     Номер кодировки Wiegand.
 * @param[out] pConfigOk    `ILR_TRUE`, конфигурация Temic правильная. Может быть равно `nullptr`.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_decode_temic_hid(const uint32_t* pData4, size_t nBlockCount,
                                                 ilr_card_uid* pUID, int* pWiegand,
                                                 ilr_bool* pConfigOk = nullptr);

/**
 * @brief Отменяет команду. Устанавливает статус `ILR_E_ABORT`.
 *
 * @param[in] hCommand  Дескриптор команды.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_command_cancel(ilr_handle hCommand);

/**
 * @brief Возвращает состояние команды.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pStatus   Состояние команды: =`ILR_E_PENDING` команда ещё выполняется, иначе -
 * завершена.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_command_get_status(ilr_handle hCommand, ilr_status* pStatus);

/**
 * @brief Возвращает состояние прогресса выполнения команды.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pCurrent  Текущий шаг.
 * @param[out] pTotal    Всего шагов.
 *
 * @return `ILR_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILR_API ilr_status ILR_CALL ilr_command_get_progress(ilr_handle hCommand, size_t* pCurrent,
                                                     size_t* pTotal);

#endif  // !ILREADERS_LINKONREQUEST

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif  // ILREADERS_H_INCLUDED
