#include "CMfPlusKeysDialog.h"

#include "utils.h"

#define HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY GTKMM_CHECK_VERSION(4, 9, 1)

CMpKeysDialog::CMpKeysDialog() :
    m_fCurrentKeyValid(false),
    m_fAccept(false),
    m_fModifed(false),
    m_VBox(Gtk::Orientation::VERTICAL),
    m_pNewKeyLabel(nullptr),
    m_pCommentLabel(nullptr),
    m_SortButton("Сортировать"),
    m_CreateButton("Создать"),
    m_DeleteButton("Удалить"),
    m_CancelButton("Отмена"),
    m_OkButton("OK") {
    set_destroy_with_parent(true);

    set_title("Ключи аутентификации Mifare Plus");
    set_default_size(550, 300);

    m_VBox.set_margin(5);
    m_VBox.set_expand();
    set_child(m_VBox);

    m_VBox.append(m_ScrolledWindow);
    m_VBox.append(m_BottomBox);

    m_ScrolledWindow.set_child(m_ColumnView);
    m_ScrolledWindow.set_expand();
    m_ListStore = Gio::ListStore<ModelColumns>::create();

    // Set list model and selection model.
    auto selection_model = Gtk::MultiSelection::create(m_ListStore);
    m_ColumnView.set_model(selection_model);
    m_ColumnView.add_css_class("data-table");  // high density table

    // Add the ColumnView's columns:

    // Столбец "№"
    auto factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_setup_label));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_bind_idx));
    auto column = Gtk::ColumnViewColumn::create("№", factory);
    m_ColumnView.append_column(column);

    // Столбец "Ключ"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_setup_key));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_bind_key));
    factory->signal_unbind().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_unbind_key));
    column = Gtk::ColumnViewColumn::create("Ключ", factory);
    m_ColumnView.append_column(column);

    // Столбец "Комментарий"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_setup_comment));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_bind_comment));
    factory->signal_unbind().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_unbind_comment));
    column = Gtk::ColumnViewColumn::create("Комментарий", factory);
    m_ColumnView.append_column(column);

    // Load extra CSS file.
    m_refCssProvider = Gtk::CssProvider::create();
#if HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY
    Gtk::StyleProvider::add_provider_for_display(get_display(), m_refCssProvider,
                                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#else
    Gtk::StyleContext::add_provider_for_display(get_display(), m_refCssProvider,
                                                GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#endif
    std::string sCSS =
        ".key {"
        "  font-family:monospace; "
        "}";
    m_refCssProvider->load_from_data(sCSS);

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.set_hexpand();
    m_BottomBox.set_halign(Gtk::Align::END);
    m_BottomBox.append(m_SortButton);
    m_BottomBox.append(m_CreateButton);
    m_BottomBox.append(m_DeleteButton);
    m_BottomBox.append(m_CancelButton);
    m_BottomBox.append(m_OkButton);

    m_SortButton.signal_clicked().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_button_sort));
    m_CreateButton.signal_clicked().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_button_create));
    m_DeleteButton.signal_clicked().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_button_delete));
    m_CancelButton.set_halign(Gtk::Align::END);
    m_OkButton.set_halign(Gtk::Align::END);
    m_CancelButton.signal_clicked().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_button_cancel));
    m_OkButton.signal_clicked().connect(sigc::mem_fun(*this, &CMpKeysDialog::on_button_ok));

    set_default_widget(m_OkButton);
}

CMpKeysDialog::~CMpKeysDialog() {
}

void CMpKeysDialog::Init(Glib::RefPtr<CMfPlusKeysSettings> refSets,
                         Glib::RefPtr<CMifareReaderSettings> refRdSets) {
    m_refSets = refSets;
    m_refRdSets = refRdSets;
    m_fModifed = false;
    UpdateView();
    if (m_fCurrentKeyValid) {
        auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
        auto nIdx = FindKey(m_CurrentKey);
        if (nIdx != GTK_INVALID_LIST_POSITION)
            pModel->select_item(nIdx, true);
    }
}

void CMpKeysDialog::on_setup_label(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    list_item->set_child(*Gtk::make_managed<Gtk::Label>());
}

void CMpKeysDialog::on_setup_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto pLabel = Gtk::make_managed<Gtk::EditableLabel>();
    list_item->set_child(*pLabel);
    pLabel->add_css_class("key");
    pLabel->property_editing().signal_changed().connect(sigc::bind(
        sigc::mem_fun(*this, &CMpKeysDialog::on_key_label_editing_change), list_item.get()));
}

void CMpKeysDialog::on_setup_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto pLabel = Gtk::make_managed<Gtk::EditableLabel>();
    list_item->set_child(*pLabel);
    pLabel->set_editable(false);
    pLabel->property_editing().signal_changed().connect(sigc::bind(
        sigc::mem_fun(*this, &CMpKeysDialog::on_comment_label_editing_change), list_item.get()));
}

void CMpKeysDialog::on_bind_idx(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_label(Glib::ustring::format(list_item->get_position()));
}

void CMpKeysDialog::on_bind_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (!col->m_property_key.get_value().m_fValid)
        m_pNewKeyLabel = label;
    Glib::Binding::bind_property_value(col->m_property_key.get_proxy(), label->property_text(),
                                       Glib::Binding::Flags::SYNC_CREATE,
                                       sigc::mem_fun(*this, &CMpKeysDialog::key_transform_to),
                                       sigc::mem_fun(*this, &CMpKeysDialog::key_transform_from));
}

void CMpKeysDialog::on_unbind_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (label == m_pNewKeyLabel)
        m_pNewKeyLabel = nullptr;
}

bool CMpKeysDialog::key_transform_to(const GValue* a, GValue* b) {
    auto pSrc = (const my_key*)a->data->v_pointer;
    if (b->g_type != G_TYPE_STRING)
        return false;
    if (pSrc->m_fValid)
        g_value_set_string(b, ilr::CMifarePlusKey(pSrc->m_key).ToString(" ").c_str());
    else
        g_value_set_string(b, "");
    return true;
}

bool CMpKeysDialog::key_transform_from(const GValue* a, GValue* b) {
    if (a->g_type != G_TYPE_STRING)
        return false;
    auto pDest = (my_key*)b->data->v_pointer;
    auto pValue = g_value_get_string(a);
    ilr::CMifarePlusKey key;
    if ('\0' == *pValue) {
        pDest->m_fValid = false;
        return true;
    }
    else if (key.TryParse(pValue)) {
        pDest->m_key = key;
        pDest->m_fValid = true;
        return true;
    }
    return false;
}

void CMpKeysDialog::on_key_label_editing_change(Gtk::ListItem* list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (!label->get_editing()) {
        my_key oNewKey;
        oNewKey.m_fValid = true;
        if (oNewKey.m_key.TryParse(label->get_text().c_str())) {
            if ((col->m_property_key.get_value().m_fValid != oNewKey.m_fValid) ||
                (col->m_property_key.get_value().m_key != oNewKey.m_key)) {
                col->m_property_key = oNewKey;
                m_fModifed = true;
                if (m_pCommentLabel != nullptr)
                    m_pCommentLabel->set_editable();
                auto nCount = m_ListStore->get_n_items();
                auto item = m_ListStore->get_item(nCount - 1);
                if (item->m_property_key.get_value().m_fValid) {
                    m_ListStore->append(
                        ModelColumns::create(my_key(false, ilr::CMifarePlusKey()), ""));
                }
            }
        }
    }
}

void CMpKeysDialog::on_bind_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    m_pCommentLabel = label;
    auto fValidKey = col->m_property_key.get_value().m_fValid;
    label->set_editable(fValidKey);
    Glib::Binding::bind_property(col->m_property_comment.get_proxy(), label->property_text(),
                                 Glib::Binding::Flags::SYNC_CREATE);
}

void CMpKeysDialog::on_unbind_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    m_pCommentLabel = nullptr;
}

void CMpKeysDialog::on_comment_label_editing_change(Gtk::ListItem* list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (!label->get_editing()) {
        auto text = label->get_text();
        if (col->m_property_comment.get_value() != text) {
            col->m_property_comment.set_value(text);
            m_fModifed = true;
        }
    }
}

void CMpKeysDialog::on_button_sort() {
    m_ListStore->sort(sigc::mem_fun(*this, &CMpKeysDialog::on_sort_compare));
    m_fModifed = true;
}

int CMpKeysDialog::on_sort_compare(const Glib::RefPtr<const ModelColumns>& a,
                                   const Glib::RefPtr<const ModelColumns>& b) {
    if (a->m_property_key.get_value().m_fValid != b->m_property_key.get_value().m_fValid)
        return a->m_property_key.get_value().m_fValid ? 1 : -1;
    if (a->m_property_key.get_value().m_fValid)
        return a->m_property_key.get_value().m_key.Compare(b->m_property_key.get_value().m_key);
    return 0;
}

void CMpKeysDialog::on_button_delete() {
    auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
    auto nCount = pModel->get_n_items();
    auto pSelection = pModel->get_selection(0, nCount);
    if (pSelection->is_empty())
        return;

    int nOffs = 0;
    for (auto it = pSelection->cbegin(); it != pSelection->cend(); it++) {
        auto nIdx = (*it - nOffs);
        auto item = m_ListStore->get_item(nIdx).get();
        if (!item->m_property_key.get_value().m_fValid)
            continue;
        m_ListStore->remove(nIdx);
        ++nOffs;
    }
    m_fModifed = true;
}

void CMpKeysDialog::on_button_create() {
    if (m_pNewKeyLabel != nullptr)
        m_pNewKeyLabel->start_editing();
}

void CMpKeysDialog::on_button_cancel() {
    set_visible(false);
}

void CMpKeysDialog::on_button_ok() {
    SaveViewData();
    m_fAccept = true;
    set_visible(false);
}

void CMpKeysDialog::UpdateView() {
    m_ListStore->remove_all();

    for (auto& rInfo : m_refSets->m_oKeys)
        m_ListStore->append(ModelColumns::create(my_key(true, rInfo.m_Key), rInfo.m_sComment));

    m_ListStore->append(ModelColumns::create(my_key(false, {}), ""));
}

void CMpKeysDialog::SaveViewData() {
    if (m_fModifed) {
        CMpKeyList oKeys;
        CMpKeyInfo rInfo;
        auto nCount = m_ListStore->get_n_items();
        for (size_t i = 0; i < nCount; i++) {
            auto item = m_ListStore->get_item(i);
            auto Key = item->m_property_key.get_value();
            if (!Key.m_fValid)
                continue;
            rInfo.m_Key = Key.m_key;
            rInfo.m_sComment = item->m_property_comment.get_value();
            oKeys.emplace_back(std::move(rInfo));
        }
        m_refSets->m_oKeys = std::move(oKeys);
        m_refSets->SetModifiedFlag();
        m_refSets->Save(GetMpKeysFilePath());
        m_fModifed = false;
    }
    auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
    auto pSelection = pModel->get_selection(0, pModel->get_n_items());
    m_fCurrentKeyValid = false;
    if (!pSelection->is_empty()) {
        auto r = m_ListStore->get_item(*pSelection->cbegin())->m_property_key.get_value();
        if (r.m_fValid)
            m_CurrentKey = r.m_key;
    }
}

guint CMpKeysDialog::FindKey(const ilr::CMifarePlusKey& key) const {
    auto nCount = m_ListStore->get_n_items();
    for (guint i = 0; i < nCount; i++) {
        auto r = m_ListStore->get_item(i)->m_property_key.get_value();
        if (r.m_fValid && (r.m_key == key))
            return i;
    }
    return GTK_INVALID_LIST_POSITION;
}
